# -*- coding: utf-8 -*-

from datetime import datetime, timedelta
import csv
import decimal
import logging
import requests
import os
import argparse

logger = logging.getLogger(__name__)

try:
    from sandbox import sdk2
    from sandbox.common.types import resource as ctr

    from sandbox.projects.common import binary_task

    class HypercubeStatParameters(sdk2.Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('Hypercube build statistics parameters') as hypercube_stat_param:
            database_host = sdk2.parameters.String(
                'database hostname',
                default='c-aeaca157-e7c8-432e-8b84-cd9fa06fbf5a.rw.db.yandex.net', )
            database_password_secret = sdk2.parameters.YavSecret(
                "database password secret",
                default='sec-01dy4hb0ngwt03am1ym70b4kjp')
            staff_api_token = sdk2.parameters.YavSecret(
                "staff api token",
                default='sec-01dy4hb0ngwt03am1ym70b4kjp'
            )
            rebuild_reports = sdk2.parameters.Bool(
                "Erase previously built data",
                default=False
            )

    class HypercubeStatistic(binary_task.LastBinaryTaskRelease, sdk2.Task):
        class Parameters(HypercubeStatParameters):
            pass

        class Requirements(sdk2.Requirements):
            client_tags = 'USER_MONOREPO'
            cores = 2
            ram = 8192

            class Caches(sdk2.Requirements.Caches):
                pass  # means that task do not use any shared caches

        @property
        def binary_executor_query(self):
            return {
                "attrs": {"task_type": "HYPERCUBE_STATISTICS",
                          "released": self.Parameters.binary_executor_release_type,
                          "target": "hypercube_statistics/bin"},
                "state": [ctr.State.READY],
                "owner": "MOBDEVTOOLS"
            }

        def on_execute(self):
            logger.info('Task started.')

            host = self.Parameters.database_host
            secret_data = self.Parameters.database_password_secret.data()
            password = secret_data["POSTGRES_PASS"]
            staff_token = secret_data["STAFF_AUTH_TOKEN"]
            rebuild_report = self.Parameters.rebuild_reports

            logger.info("Parameters: {}".format(self.Parameters))
            report_builder = BuildReport(host, password, staff_token)
            if rebuild_report:
                report_builder.clear_report()
            report_builder.build_report()

            logger.info('Task finished.')

except:
    logger.info("Program is executed locally, not in Sandbox")


class Staff:
    def __init__(self, token):
        pass


class BuildReport:
    def __init__(self, host, password, staff_token):
        self.pg = PG(host, password)
        self.calc = Calculate()

    def clear_report(self):
        self.pg.clear_all_reports()

    def build_report(self):

        self.pg.update_first_connect_date()

        logger.info("Build report since {}".format(self.pg.last_calculated_date))
        cur_date = self.pg.last_calculated_date

        while cur_date.date() < datetime.today().date():
            current_owners = []
            ts_midnight = (cur_date - datetime(1970, 1, 1)).total_seconds()
            next_date = cur_date + timedelta(days=1)
            ts_next_midnight = (next_date - datetime(1970, 1, 1)).total_seconds()

            initial_owners = self.pg.get_owners(for_date=cur_date)
            destroyed_devices = self.pg.get_destroyed_devices()
            period_events = self.pg.get_events(period_start=ts_midnight,
                                               period_end=ts_next_midnight)
            daily_devices_cost = {}

            for deviceid, purchase_price, purchase_date, first_connect_date in self.pg.devices:
                logger.debug("Build report for device {}".format(deviceid))

                # устройство еще не было подключено к кубику в обрабатываемый период? пропускаем
                if first_connect_date > cur_date.date():
                    continue

                # устройство уже списано к текущему моменту? пропускаем
                if deviceid in destroyed_devices and destroyed_devices[deviceid] < ts_midnight:
                    logger.debug("Skipping destroyed device {} on {}".format(deviceid, destroyed_devices[deviceid]))
                    continue

                # получаем владельца устройства на начало обрабатываемого периода
                if deviceid in initial_owners:
                    initial_owner = initial_owners[deviceid]
                else:
                    initial_owner = 'UNKNOWN'

                # filter specific device events from all events
                device_events = [(e_ts, e_operator, e_location) for e_ts, e_operator, e_location, e_device_id in
                                 period_events if e_device_id == deviceid]

                # calculate rental cost for device at designated period
                device_report, last_owner = self.calc.period_cost(initial_owner=initial_owner,
                                                                  device_events=device_events,
                                                                  period_start=ts_midnight,
                                                                  period_end=ts_next_midnight,
                                                                  purchase_price=purchase_price,
                                                                  purchase_date=purchase_date,
                                                                  )

                logger.debug(device_report)
                logger.debug("Date: {}. Device {}. Report: {}. Owner: {}".format(cur_date,
                                                                                 deviceid,
                                                                                 device_report,
                                                                                 last_owner))
                # remember device owner at period_end
                current_owners.append((next_date, deviceid, last_owner))

                # summarize rental cost for operator
                for operator, cost in device_report.items():
                    if operator in daily_devices_cost:
                        daily_devices_cost[operator] += cost
                    else:
                        daily_devices_cost[operator] = cost

            logger.debug("Date: {}. current_owners: {}".format(next_date, current_owners))
            self.pg.append_owners(current_owners)

            self.pg.append_report(
                [(cur_date, operator, cost) for operator, cost in daily_devices_cost.items() if cost > 0])
            logger.debug("Date: {}. Daily cost {}".format(cur_date, daily_devices_cost))
            cur_date = next_date


class PG:
    pg_connection_string = """
        host={}
        port=6432
        sslmode=verify-full
        dbname=tesseract
        user=tesseract
        password={}
        target_session_attrs=read-write
        sslrootcert=psql_root.crt
    """

    def __init__(self, host, password):
        self._setup_connect(self.pg_connection_string.format(host, password))
        self.operators = self.get_operators()
        self.devices = self.get_devices()
        last_calc_date = self._get_last_calculated_date()

        # first legitimate event ts - 1499361689.05702  2017-07-06 20:21:29.05702+03
        if last_calc_date[0] is None:
            self.last_calculated_date = datetime.strptime('2017-07-06 00:00:00', '%Y-%m-%d %H:%M:%S')
        else:
            self.last_calculated_date = last_calc_date[0]

    @staticmethod
    def _prepare_pg_connect():
        req = requests.get("https://crls.yandex.net/allCAs.pem", allow_redirects=True)
        pg_cert = 'psql_root.crt'
        with open(pg_cert, "w+b") as file_crt:
            file_crt.write(req.content)
        logger.info("Created {}".format(pg_cert))

    def _setup_connect(self, config):
        self._prepare_pg_connect()
        import psycopg2
        self.conn = psycopg2.connect(config)
        self.cursor = self.conn.cursor()

    def _fetch_all_data(self, query):
        self.cursor.execute(query)
        return self.cursor.fetchall()

    def _fetch_one_line(self, query):
        self.cursor.execute(query)
        return self.cursor.fetchone()

    def get_destroyed_devices(self):
        query = '''
            SELECT device_id, ts
            FROM events
            WHERE location_id = 'destroyed' AND is_last = True
            '''
        return dict(self._fetch_all_data(query))

    def clear_all_reports(self):
        query = "DELETE FROM calculated_cost; DELETE FROM midnight_owners;"
        self.cursor.execute(query)

    def get_devices(self):
        query = "SELECT id, purchase_price, purchase_date, first_connect_date FROM devices;"
        return self._fetch_all_data(query)

    def get_operators(self):
        query = "SELECT DISTINCT(operator) from events"
        return self._fetch_all_data(query)

    def get_owners(self, for_date):
        query = "SELECT device_id, operator FROM midnight_owners WHERE ts = \'{}\'"
        return dict(self._fetch_all_data(query.format(for_date)))

    def get_events(self, period_start, period_end):
        query = '''
            SELECT ts, operator, location_id, device_id
            FROM events
            INNER JOIN tesseracts ON events.tesseract_id = tesseracts.id
            WHERE ts >= {} AND ts < {}
            ORDER BY ts'''.format(period_start, period_end)

        return self._fetch_all_data(query)

    def append_owners(self, data):
        # TODO implement bulk UPSERT
        # INSERT INTO midnight_owners (ts, device_id, operator) VALUES %s ON CONFLICT ON CONSTRAINT uniq_ts_device_id DO UPDATE SET operator = 'xxx'

        query = '''
            INSERT INTO midnight_owners (ts, device_id, operator) VALUES %s ;
        '''
        from psycopg2.extras import execute_values
        execute_values(self.cursor, query, data)
        self.conn.commit()

    def append_report(self, data):
        query = '''
            INSERT INTO calculated_cost (ts, operator, cost) VALUES %s ;
        '''
        from psycopg2.extras import execute_values
        execute_values(self.cursor, query, data)
        self.conn.commit()

    def _get_last_calculated_date(self):
        query = '''
            SELECT MAX(ts) FROM midnight_owners
        '''
        return self._fetch_one_line(query)

    def update_devices(self, data):
        """
        Update table devices from data list ('inventory', 'serial', 'price', 'purchase_date')
        """

        query_update_date = """UPDATE devices
                               SET serial = update_payload.serial,
                               purchase_price = update_payload.price,
                               purchase_date = update_payload.purchase_date

                               FROM (VALUES %s) AS update_payload (inventory, serial, price, purchase_date)
                               WHERE devices.inventory = update_payload.inventory
                               """
        from psycopg2.extras import execute_values
        execute_values(self.cursor, query_update_date, data)
        self.conn.commit()

    def update_first_connect_date(self):
        """
        Fill `first_connect_date` field in `devices` table
        """

        query_unhandled_devices = '''SELECT id FROM devices WHERE first_connect_date IS NULL'''
        # "ts >= 1499361689" - это условие пришлось добавить т.к есть события с фейковой датой 2001-* и 2016-*,
        # появившиеся из-за того что RPI не имеет энергонезависимую память.
        query_get_first_date = '''SELECT ts FROM events WHERE device_id = '{}' and ts >= 1499361689 ORDER BY ts LIMIT 1'''
        query_update_date = """UPDATE devices
                               SET first_connect_date = update_payload.first_connect_date
                               FROM (VALUES %s) AS update_payload (id, first_connect_date)
                               WHERE devices.id = update_payload.id
                               """

        devices = self._fetch_all_data(query_unhandled_devices)
        data = []
        for device in devices:
            id = device[0]
            ts = self._fetch_one_line(query_get_first_date.format(id))

            # don't ask why, but there are devices without any events.
            if ts is not None:
                first_connect_date = datetime.fromtimestamp(ts[0]).date()
            else:
                first_connect_date = datetime.strptime('2000-01-01', '%Y-%m-%d').date()
            data.append((id, first_connect_date))

        from psycopg2.extras import execute_values
        execute_values(self.cursor, query_update_date, data)
        self.conn.commit()


class Calculate:
    def __init__(self):
        self.amortized_years = 4
        self.rate = self._set_double_declining_balance_rate(self.amortized_years)
        self._TWOPLACES = decimal.Decimal(10) ** -2

    @staticmethod
    def _set_double_declining_balance_rate(amortized_years):
        rate = {}
        BV = 1
        SLDP = float(1.0 / amortized_years)
        for i in range(amortized_years):
            rate[i] = 2 * SLDP * BV
            BV -= rate[i]

        #  rate sum must be 100%. i.e for 4 year rates: 0.5, 0.25, 0.125, 0.125
        rate[amortized_years - 1] = rate[amortized_years - 2]
        return rate

    def get_rate(self, purchase_date, current_date):
        # если дата события меньше даты покупки - считаем стоимость аренды нулевой.
        # если дата события больше даты окончания амортизационного периода - тоже считаем стоимость нулевой
        # purchase_date всегда хранится с 00:00:00, поэтому не нужно обрабатывать кейс когда
        # в обрабатываемых сутках, часть времени что-то стоит, а часть уже нет.

        # code doesn't count leap years properly
        years_count = (current_date - purchase_date).days // 365
        if years_count < 0 or years_count > self.amortized_years - 1:
            return 0
        else:
            return self.rate[years_count]

    def period_cost(self, initial_owner, device_events, period_start, period_end, purchase_price,
                    purchase_date):
        """
        period_start - timestamp начала расчетных суток
        period_end - timestamp конца суток
        Return:
        report: dict [operator] - cost.
        last owner: name of the last device owner in a calculated day
        """

        report = {}
        '''
            operator    |  location_id  |         |        ts        | purchase_price | purchase_date | group_id
        ----------------+---------------+------------------+------------------+----------------+---------------+----------
         netimen        | tesseract     |  | 1551870701.03645 |              0 |               | mobile
         syntezzz       | operator      |  | 1551881193.06306 |              0 |               | mobile
         syntezzz       | tesseract     |  | 1551885145.12616 |              0 |               | mobile
         imkozhin       | operator      |  | 1551885924.28216 |              0 |               | mobile
         imkozhin       | expired       |  |    1552633203.64 |              0 |               | mobile

              id       |                        name
        ---------------+-----------------------------------------------------
         tesseract     | В шкафу
         unreserved    | Бронь отменена
         reserved      | Зарезервировано
         operator      | На руках
         takeAway      | Отобрали устройство
         expired       | Истёк срок пользования
         prolonged     | Продлено
         service       | В ремонте
         doorOpen      | Дверь шкафа открыта
         doorClose     | Дверь шкафа закрыта
         giveAway      | Передали из рук в руки
         destroyed     | Выведено из эксплуатации
         inventory     | Инвентаризуется
         undefined     | Местонахождение неизвестно
         longterm      | В долгосрочной аренде
         remoteSession | Используется для удаленного доступа сервисом Колхоз
        '''

        current_period_start = period_start
        current_operator = initial_owner
        if purchase_date is None:
            purchase_date = datetime.strptime('2000-01-01', '%Y-%m-%d').date()

        start_date = datetime.fromtimestamp(period_start).date()
        cost_per_minute = float(purchase_price) * self.get_rate(purchase_date, start_date) / (365 * 24 * 60)

        for (current_period_end, operator, location) in device_events:
            if location in ('operator', 'takeAway', 'longterm'):
                next_operator = operator
            # по всем остальным событиям считаем что владелец устройства - гиперкуб.
            # если устройство передают из рук в руки, то
            # - после записи giveAway считаем что устройство взял гиперкуб
            # - после записи takeAway начнем считать что устройство взял пользователь.
            else:
                next_operator = 'tesseract'

            self._append_report(report, cost_per_minute, current_period_start, current_period_end, current_operator)
            current_operator = next_operator
            current_period_start = current_period_end

        # считаем последний участок (либо все сутки если с девайсом не было никаких событий)
        self._append_report(report, cost_per_minute, current_period_start, period_end, current_operator)
        return report, current_operator

    def _append_report(self, report, cost_per_minute, start, end, operator):
        rental_cost = decimal.Decimal(cost_per_minute * (float(end - start) / 60)).quantize(self._TWOPLACES)
        if rental_cost == decimal.Decimal(0.00):
            return
        if operator not in report:
            report[operator] = rental_cost
        else:
            report[operator] = decimal.Decimal(report[operator] + rental_cost).quantize(self._TWOPLACES)


class FillData:
    def __init__(self, pg, filename):
        self.read_data(filename)
        self.write_data(pg)

    def read_data(self, filename):
        update_data = []
        _TWOPLACES = decimal.Decimal(10) ** -2
        with open(filename, 'r', encoding='cp1251') as fh:
            reader = csv.reader(fh, dialect='excel', delimiter='\t')
            reader.__next__()
            for row in reader:
                logger.info("Row {}".format(row))
                inventory = row[0]
                serial = row[3]
                raw_price = row[-1]
                if raw_price != '':
                    price = raw_price.split()[0].replace('.', '').replace(',', '.')
                else:
                    price = decimal.Decimal(0)
                purchase_date = datetime.strptime(row[10], '%d.%m.%Y').date()
                # до 2019.01.01 НДС = 18%, с 2019.01.01 НДС = 20%
                if purchase_date.year < 2019:
                    price = decimal.Decimal(float(price) * 1.18).quantize(_TWOPLACES)
                else:
                    price = decimal.Decimal(float(price) * 1.2).quantize(_TWOPLACES)
                update_data.append((inventory, serial, price, purchase_date))
        self.new_data = update_data
        print('Read data: {}', self.new_data)

    def write_data(self, pg):
        pg.update_devices(self.new_data)


if __name__ == "__main__":
    logger.info('Task started.')
    parser = argparse.ArgumentParser()
    parser.add_argument('-t', '--pg-host', default='c-aeaca157-e7c8-432e-8b84-cd9fa06fbf5a.rw.db.yandex.net')
    parser.add_argument('-p', '--pg-pass', default='~/.pgaas_secret')
    parser.add_argument('-s', '--staff-token', default='~/.staff_secret')
    parser.add_argument('-r', '--rebuild-report', default=0)
    parser.add_argument('-d', '--devices', default='')

    args = parser.parse_args()
    host = args.pg_host
    if args.pg_pass != '':
        with open(os.path.expanduser(args.pg_pass), 'r') as fh:
            password = fh.readline().strip()
    if args.staff_token != '':
        with open(os.path.expanduser(args.staff_token), 'r') as fh:
            staff_token = fh.readline().strip()
    rebuild_report = bool(args.rebuild_report)
    device_datafile = os.path.expanduser(args.devices)
    pg = PG(host, password)

    if device_datafile != '':
        FillData(pg, device_datafile)

    c = Calculate()
    reportBuilder = BuildReport(host, password, staff_token)
    if rebuild_report:
        reportBuilder.clear_report()
    reportBuilder.build_report()

    logger.info('Task finished.')
