import os
import logging
import shutil

from sandbox import sdk2
from sandbox.common.types import resource as ctr
from sandbox.projects.common import binary_task
from sandbox.sdk2.vcs.git import Git
from sandbox.projects.mobile_apps.utils.resources import ProvisionProfile

logger = logging.getLogger(__name__)


class ProvisionProfilePreparerParameters(sdk2.Task.Parameters):
    ext_params = binary_task.binary_release_parameters(stable=True)

    with sdk2.parameters.Group('provision profile updater parameters') as params:
        ssh_key = sdk2.parameters.Vault(
            'SSH key', )


class ProvisionProfilePreparer(binary_task.LastBinaryTaskRelease, sdk2.Task):
    bitbucket_repo_url = 'ssh://git@bitbucket.browser.yandex-team.ru/mt/mobile-ios-provisioning-profiles.git'
    bitbucket_provision_profiles_dirs = ['AppStore', 'AdHoc']
    bitbucket_hash = ''
    profiles_folder = 'Provisioning_Profiles'

    class Parameters(ProvisionProfilePreparerParameters):
        pass

    class Requirements(sdk2.Requirements):
        client_tags = 'USER_MONOREPO'
        cores = 2
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    @property
    def binary_executor_query(self):
        return {
            "attrs": {"task_type": "PROVISION_PROFILE_PREPARER",
                      "released": self.Parameters.binary_executor_release_type,
                      "target": "provision_profile_preparer/bin"},
            "state": [ctr.State.READY],
            "owner": "MOBDEVTOOLS"
        }

    def on_execute(self):
        logger.info('Task started.')
        self.git_clone()
        if not self.is_resource_already_exist():
            self.create_folder_with_profiles()
            self.create_resource()
        logger.info('Task finished.')

    def git_clone(self):
        logger.info("Cloning repo {}".format(self.bitbucket_repo_url))
        with sdk2.ssh.Key(self, self.Parameters.ssh_key.owner, self.Parameters.ssh_key.name):
            git = Git(self.bitbucket_repo_url, filter_branches=False)
            git.clone('.', "refs/heads/master")
            git_hash_proc = git.raw_execute(["rev-parse", "HEAD"])
            stdout, stderr = git_hash_proc.communicate()
            self.bitbucket_hash = stdout[:10]
            if len(self.bitbucket_hash) == 0:
                raise ValueError("Got empty hash from bitbucket")
        logger.info("Cloned repo")

    def is_resource_already_exist(self):
        attrs = {"hash": self.bitbucket_hash}
        resource = sdk2.Resource.find(type="PROVISION_PROFILE", attrs=attrs).first()
        if not resource or resource.state != "READY":
            logger.info("Resource with hash {} not found is Sandbox".format(self.bitbucket_hash))
            return False
        else:
            logger.info(
                "Resource with hash {} is already present in Sandbox - {}".format(self.bitbucket_hash, resource.id))
            return True

    def create_folder_with_profiles(self):
        logger.info("Copy files from 'AppStore', 'AdHoc to {}".format(self.profiles_folder))
        os.makedirs(self.profiles_folder)
        for directory in self.bitbucket_provision_profiles_dirs:
            all_profiles = [f for f in os.listdir(directory) if os.path.isfile(os.path.join(directory, f))]
            for profile in all_profiles:
                shutil.copyfile(os.path.join(directory, profile),
                                os.path.join(self.profiles_folder, profile))
        logger.info("Copied files")

    def create_resource(self):
        attrs = {"hash": self.bitbucket_hash}
        description = 'ProvisionProfile, hash {}'.format(self.bitbucket_hash)
        resource = ProvisionProfile(self, description, path=self.profiles_folder, **attrs)
        sdk2.ResourceData(resource).ready()
        logger.info('New resource with attributes {} was generated: {}'.format(attrs, resource.id))
