import logging
import os

from sandbox import sdk2
from sandbox.common.types import client as ctc
from sandbox.common.types import misc as ctm
from sandbox.common.errors import TaskFailure
from sandbox.projects.common import binary_task
from sandbox.projects.mobile_apps.utils.resources import RvmPlusRuby
from sandbox.projects.mobile_apps.utils import shellexecuter
from sandbox.sandboxsdk.environments import Xcode

_logger = logging.getLogger(__name__)

RVM_VERSION = '1.29.10'
RUBY_VERSION = '2.6.5'
# The simplest way to install is to run the entire script in a single file
# since rvm sets environment variables that are easier to live in one process
RVM_CMD_MACOS = '''
#!/bin/bash
if which gpg >/dev/null; then
    curl -sSL https://rvm.io/mpapis.asc | gpg --import -
    curl -sSL https://rvm.io/pkuczynski.asc | gpg --import -
fi
curl -sSL https://get.rvm.io | bash -s {} --ignore-dotfiles

wget --no-check-certificate https://github.com/openssl/openssl/archive/refs/tags/OpenSSL_1_1_1l.tar.gz
tar -xf OpenSSL_1_1_1l.tar.gz
cd openssl-OpenSSL_1_1_1l
./config --prefix=$HOME/.rvm/usr
make install_sw

cd $HOME
curl -s https://ftp.gnu.org/gnu/readline/readline-8.1.tar.gz -o readline-8.1.tar.gz
tar -xf readline-8.1.tar.gz
cd readline-8.1
./configure --prefix=$HOME/.rvm/usr
make install

source $HOME/.rvm/scripts/rvm
export PKG_CONFIG_PATH=$rvm_path/usr/lib/pkgconfig
export LDFLAGS="-L$rvm_path/usr/lib"
rvm autolibs disable
rvm install {}

rvm all do ruby -ropenssl -e 'puts OpenSSL::OPENSSL_VERSION'
rvm all do ruby -ropenssl -e 'puts OpenSSL::OPENSSL_LIBRARY_VERSION'
rvm all do ruby -r rbconfig -e 'puts RbConfig::CONFIG["configure_args"]'

printenv
'''

RVM_CMD_LINUX = '''
#!/bin/bash
if which gpg >/dev/null; then
    curl -sSL https://rvm.io/mpapis.asc | gpg --import -
    curl -sSL https://rvm.io/pkuczynski.asc | gpg --import -
fi
curl -sSL https://get.rvm.io | bash -s {} --ignore-dotfiles
source $HOME/.rvm/scripts/rvm
export PKG_CONFIG_PATH=$rvm_path/usr/lib/pkgconfig
rvm autolibs disable
rvm pkg install openssl
rvm install {} --with-openssl-dir=$rvm_path/usr
'''


class RvmPlusRubyPreparerParameters(sdk2.Task.Parameters):

    ext_params = binary_task.binary_release_parameters(stable=True)

    with sdk2.parameters.Group('rvm+ruby parameters') as params:
        rvm_version = sdk2.parameters.String(
            'RVM version',
            default=None, )
        ruby_version = sdk2.parameters.String(
            'Ruby version',
            default=None, )
        with sdk2.parameters.RadioGroup("Build resource on platform") as platform:
            platform.values["linux_x86"] = platform.Value("Linux")
            platform.values["macos_x86"] = platform.Value("Macos, Intel", default=True)
            platform.values["macos_arm"] = platform.Value("Macos, ARM")
        xcode_version = sdk2.parameters.String(
            "Xcode version to install. Applicable to macos_* platforms only",
            default="12.5.1",
        )


class RvmPlusRubyPreparer(sdk2.Task):

    #  for 'macos_arm' choose binary 'bin(macos_x86)', because sandbox runs with Rosetta
    # 'macos_arm', 'macos_x86', 'linux_x86' are used as resource's platform and arch attributes
    platform_parameters = {"linux_x86": {"target": "rvm_plus_ruby_preparer/bin",
                                         "client_tags": "USER_MONOREPO & LINUX",
                                         },
                           "macos_x86": {"target": "rvm_plus_ruby_preparer/bin(macos_x86)",
                                         "client_tags": "USER_MONOREPO & MOBILE_MONOREPO",
                                         },
                           "macos_arm": {"target": "rvm_plus_ruby_preparer/bin(macos_x86)",
                                         "client_tags": "USER_MONOREPO & M1 & MOBILE_MONOREPO",
                                         },
                           }

    class Parameters(RvmPlusRubyPreparerParameters):
        pass

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64
        cores = 2
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    def on_save(self):
        self.Requirements.client_tags = self.platform_parameters[self.Parameters.platform]["client_tags"]
        try:
            xcode_major_version = str(self.Parameters.xcode_version).split('.')[0]
        except:
            raise RuntimeError("Failed to parse xcode version")

        if xcode_major_version in ('12', '13'):
            self.Requirements.client_tags &= ctc.Tag.OSX_BIG_SUR
        elif xcode_major_version in ('11',):
            self.Requirements.client_tags &= ctc.Tag.OSX_CATALINA

        target = self.platform_parameters[self.Parameters.platform]["target"]
        binary_resource = sdk2.service_resources.SandboxTasksBinary.find(
            owner="MOBDEVTOOLS",
            attrs={
                "task_type": "RVM_PLUS_RUBY_PREPARER",
                "target": target,
                "released": self.Parameters.binary_executor_release_type,
            }
        ).first()

        if not binary_resource:
            raise TaskFailure("Could not find binary task")
        self.Requirements.tasks_resource = binary_resource.id

    def _execute_cmd(self, args, message):
        shellexecuter.execute_on_platform(self, args, 'rvm_plus_ruby_preparing', False, message, cwd=None)

    def _prepare_rvm(self):
        self._rvm_version = self.Parameters.rvm_version or RVM_VERSION
        self._ruby_version = self.Parameters.ruby_version or RUBY_VERSION
        script_file = 'run.sh'
        RVM_CMD = RVM_CMD_LINUX if shellexecuter.detect_platform() == 'linux_x86' else RVM_CMD_MACOS
        with open(script_file, 'w') as f:
            f.write(RVM_CMD.format(self._rvm_version, self._ruby_version))
        self._execute_cmd(
            'bash -xe {}'.format(script_file),
            'Can not install rvm + ruby. See common.log.')

    def _pack_rvm_plus_ruby(self):
        user_path = os.path.expanduser('~/')
        rvm_plus_ruby_path = '.rvm'
        rvm_plus_ruby_archive_path = 'rvm.tar.gz'

        self._execute_cmd(
            'tar czf {} -C {} {}'.format(rvm_plus_ruby_archive_path, user_path, rvm_plus_ruby_path),
            'Can not pack rvm + ruby. See common.log.')
        attrs = {
            'version': '{}+{}'.format(self._rvm_version, self._ruby_version),
            'platform': self.Parameters.platform,
            'ttl': 'inf'
        }
        description = 'rvm: {} + ruby: {}'.format(self._rvm_version, self._ruby_version)
        sdk2.ResourceData(RvmPlusRuby(self, description, rvm_plus_ruby_archive_path, **attrs)).ready()

    def on_execute(self):

        if self.Parameters.platform == "macos_x86" or self.Parameters.platform == "macos_arm":
            _logger.info("Prepare xcode {}".format(self.Parameters.xcode_version))
            xcode_environment = Xcode(self.Parameters.xcode_version)
            xcode_environment.prepare()
            _logger.info("Prepared xcode")

        with self.memoize_stage.prepare_rvm:
            self._prepare_rvm()
        self._pack_rvm_plus_ruby()
