import logging
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.mobile_apps.teamcity_sandbox_runner.runner import TeamcitySandboxRunner
from sandbox.projects.mobile_apps.teamcity_sandbox_runner.utils import resources
from sandbox.projects.mobile_apps.teamcity_sandbox_runner.utils import vcs
from sandbox.projects.mobile_apps.teamcity_sandbox_runner.utils.parameters import TeamcitySandboxRunnerUpperLevelParameters
from sandbox.projects.mobile_apps.teamcity_sandbox_runner.utils.requirements import SimpleRequirements

logger = logging.getLogger('launcher')


class TeamcitySandboxRunnerLauncherParams(TeamcitySandboxRunnerUpperLevelParameters):
    pass


class TeamcitySandboxRunnerLauncher(sdk2.Task):
    class Requirements(SimpleRequirements):
        pass

    class Parameters(TeamcitySandboxRunnerLauncherParams):
        pass

    def on_save(self):
        if self.Parameters.release_type == 'custom':
            return
        if self.Parameters.custom_config_branch != '':
            self.Parameters.branch = self.Parameters.custom_config_branch
        if self.Parameters.custom_config_repo_url != '':
            self.Parameters.repo_url = self.Parameters.custom_config_repo_url
        if self.Parameters.custom_config_commit != '':
            self.Parameters.commit = self.Parameters.custom_config_commit
        if self.Parameters.custom_config_commit == 'empty':
            self.Parameters.commit = ''
        if self.Parameters.release_type == 'none':
            self.Requirements.tasks_resource = None
        else:
            binary_resource = sdk2.service_resources.SandboxTasksBinary.find(
                owner="MOBDEVTOOLS",
                attrs={
                    "target": "teamcity_sandbox_runner/bin",
                    "release": self.Parameters.release_type,
                    "tasks_bundle": "TEAMCITY_SANDBOX_RUNNER",
                }
            ).first()
            if not binary_resource:
                raise TaskFailure("Could not find binary task")
            self.Requirements.tasks_resource = binary_resource.id

    def _run_teamcity_runner(self):
        params = {}
        for key, value in self.Parameters.__iter__():
            params[key] = value
        params['version'] = self.Context.config.get('config', {}).get('runner_version', '')
        params['use_parent_resources'] = True
        # you must create messages first because of the publishing specifics
        params['teamcity_messages_resource'] = resources.create_teamcity_messages_resource(self)
        params['teamcity_artifacts_resource'] = resources.create_teamcity_artifacts_resource(self)
        if self.Context.config.get('templates'):
            params['dependent_templates'] = list(self.Context.config['templates'])
        # patch for the version when moving to Arcadia
        if self.Parameters.config_from_repository and params['version'].endswith('p') and self.Parameters.config_path.startswith('mobile/'):
            params['config_path'] = self.Parameters.config_path[7:]
        if self.Parameters.version != '':
            params['version'] = self.Parameters.version
        config_parser_sub_task = TeamcitySandboxRunner(
            self,
            description=self.Parameters.description,
            **params
        )
        config_parser_sub_task.Requirements.dns = ctm.DnsType.DNS64
        config_parser_sub_task.save()
        config_parser_sub_task.enqueue()
        self.Context.teamcity_runner_task_id = config_parser_sub_task.id

    def _check_subtask_for_success(self, task_id):
        if self.server.task[task_id].read()["status"] not in ctt.Status.Group.SUCCEED:
            raise TaskFailure('Subtask failed. More details in subtask {}.'.format(task_id))

    def on_execute(self):
        # this task is only for versioning purposes, it shouldn't do any work
        with self.memoize_stage.prelaunch_preparation:
            self.Context.config, _, _ = vcs.prepare_repository_and_read_config(
                self,
                self.Parameters.ssh_key,
                self.Parameters.repo_url,
                self.Parameters.branch,
                self.Parameters.commit,
                self.Parameters.config_from_repository,
                self.Parameters.config_path,
                self.Parameters.config,
                True
            )
            self.hint(self.Context.config.get('config').get('name'))

        with self.memoize_stage.run_teamcity_runner:
            self._run_teamcity_runner()

        with self.memoize_stage.wait_teamcity_runner:
            raise sdk2.WaitTask(self.Context.teamcity_runner_task_id, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

        self._check_subtask_for_success(self.Context.teamcity_runner_task_id)

    def _stop_task(self, task_id):
        logger.info('Try to stop task {}.'.format(task_id))
        task = sdk2.Task[task_id]
        try:
            task.stop()
        except Exception as e:
            logger.exception(e)
        logger.info('Stopped task {}.'.format(task_id))

    def _stop_child_process(self):
        self._stop_task(self.Context.teamcity_runner_task_id)

    def on_terminate(self):
        self._stop_child_process()

    def on_before_timeout(self, seconds):
        self._stop_child_process()
