import ast
import base64
import logging
import os
import shutil

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.sandboxsdk.environments import Xcode
from sandbox.sdk2 import yav
from sandbox.sdk2.helpers import subprocess
import sandbox.common.types.misc as ctm

from sandbox.projects.mobile_apps.teamcity_sandbox_runner.utils import vcs
from sandbox.projects.mobile_apps.teamcity_sandbox_runner.utils.artifacts_processor import ArtifactsProcessor
from sandbox.projects.mobile_apps.teamcity_sandbox_runner.utils.caches_preparer import CachesPreparer
from sandbox.projects.mobile_apps.teamcity_sandbox_runner.utils.parameters import TeamcitySandboxRunnerCommonParameters
from sandbox.projects.mobile_apps.teamcity_sandbox_runner.utils.resources import TeamcitySandboxRunnerInternalArtifacts
from sandbox.projects.mobile_apps.teamcity_sandbox_runner.utils.string_preparer import StringPreparer
from sandbox.projects.mobile_apps.utils import shellexecuter
from sandbox.projects.mobile_apps.utils.android_sdk_env import AndroidSdkEnvironment
from sandbox.projects.mobile_apps.utils.jdk_env import JdkEnvironment
from sandbox.projects.mobile_apps.utils.keychain_env import MacOsKeychainEnvironment
from sandbox.projects.mobile_apps.utils.provision_profiles_env import ProvisionProfileEnvironment
from sandbox.projects.mobile_apps.utils.rvm_plus_ruby_env import RvmPlusRubyEnvironment

_logger = logging.getLogger('stage')

# LOG_FORMAT = '%(asctime)s (%(delta)5.2fs) %(levelname)-6s %(name)-6s | %(message)s'
LOG_FORMAT = '%(asctime)s %(message)s'
TIME_FORMAT = '%Y-%m-%d %H:%M:%S'


class TeamcitySandboxRunnerStageParameters(TeamcitySandboxRunnerCommonParameters):
    with sdk2.parameters.Group('Teamcity sandbox runner stage parameters') as tsr_runner_params:
        name = sdk2.parameters.String(
            'Name',
            hint=True, )
        work_dir = sdk2.parameters.String(
            'Work dir',
            default='.', )
        cmd = sdk2.parameters.String(
            'Cmd', )
        caches = sdk2.parameters.String(
            'Caches',
            default='[]', )
        env = sdk2.parameters.String(
            'Env',
            default='{}', )
        secrets = sdk2.parameters.String(
            'Secrets',
            default='{}', )
        secret_files = sdk2.parameters.String(
            'Secret files',
            default='{}', )
        lxc = sdk2.parameters.Container(
            'Container',
            default=None,
            required=False, )
        artifacts = sdk2.parameters.String(
            'Artifacts',
            default=None, )
        internal_artifacts = sdk2.parameters.String(
            'Internal artifacts',
            default='{}', )
        dependency_files = sdk2.parameters.String(
            'Dependency files',
            default='{}', )
        multislot = sdk2.parameters.String(
            'Multislot', )
        ios = sdk2.parameters.Bool(
            'ios',
            default=False, )
        with ios.value[True]:
            rosetta = sdk2.parameters.Bool(
                'Execute cmd with rosetta',
                default=False,
            )
        sandbox_environments = sdk2.parameters.String(
            'Sandbox environments',
            default='{}', )
        emulator_system_images = sdk2.parameters.String(
            'System images used with emulator',
            default='',
            description='if parameter is not empty, stage will run emulator with specified images',
        )
        emulator_device_skin = sdk2.parameters.String(
            'Required emulator device skin',
            default='Nexus 5',
            description='If you want to launch mobile device you can specify skin: "Nexus 5", "pixel_5" or for TV device: "tv_1080p", "tv_720p", "tv_4k"',
        )
        emulator_disk_size = sdk2.parameters.Integer(
            'Required emulator data partition size in GB',
            default=1,
        )
        disk_space = sdk2.parameters.Integer(
            'Required disk space in GB',
            default=32,
        )
        with ios.value[False]:
            ramdisk = sdk2.parameters.Integer(
                'Required Ramdisk space in GB',
                default=2,
            )
        arc_exported_paths = sdk2.parameters.String(
            'Arcadia exported paths list',
            default='[]'
        )
        arc_export_prefix = sdk2.parameters.String(
            'export path prefix',
            default=''
        )
        config_hash = sdk2.parameters.String(
            'config hash',
            default=''
        )


class TeamcitySandboxRunnerStage(sdk2.Task):

    class Requirements(sdk2.Requirements):
        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(TeamcitySandboxRunnerStageParameters):
        pass

    def on_save(self):
        self.Requirements.disk_space = self.Parameters.disk_space * 1024
        if not self.Parameters.ios:
            self.Requirements.ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, self.Parameters.ramdisk * 1024, None)
        if self.Parameters.release_type == 'custom':
            return
        if self.Parameters.release_type == 'none':
            self.Requirements.tasks_resource = None
        else:
            attrs = {"target": "teamcity_sandbox_runner/bin",
                     "release": self.Parameters.release_type,
                     "tasks_bundle": "TEAMCITY_SANDBOX_RUNNER",
                     }
            if self.Parameters.ios:
                attrs['target'] = 'teamcity_sandbox_runner/bin(osx)'
            if self.Parameters.version != '':
                attrs['version'] = self.Parameters.version
            binary_resource = sdk2.service_resources.SandboxTasksBinary.find(
                owner="MOBDEVTOOLS",
                attrs=attrs
            ).first()
            if not binary_resource:
                raise TaskFailure("Could not find binary task")
            self.Requirements.tasks_resource = binary_resource.id

    def _init_env(self):
        """Some customer's lxc containers have predefined variables in /etc/profile.
            They must be initialized before any other operations.
        """
        _logger.info('Init enviroment. ')
        env = [l.split('=', 1) for l in
               subprocess.check_output(['/bin/bash', '-c', '. /etc/profile && printenv']).replace(',', '\n').splitlines()]
        env = [pair for pair in env if len(pair) == 2]
        self._env = dict(env)

    def _set_java_env(self):
        """
        Skip _JAVA_OPTIONS setup if it is already present in env.
        Set MaxRam, ActiveProcessorCount, java.io.tmpdir for linux only
        ADDITIONAL_JAVA_OPTIONS are added to _JAVA_OPTIONS for both linux, macos.
        """
        if '_JAVA_OPTIONS' not in self._env:
            if not self.Parameters.ios:
                self._env['_JAVA_OPTIONS'] = '-XX:MaxRAM={}G -XX:ActiveProcessorCount={} -Djava.io.tmpdir={} {}'.format(
                    self.Requirements.ram / 1024,
                    self.Requirements.cores,
                    self.ramdrive.path,
                    self._env.get('ADDITIONAL_JAVA_OPTIONS', '')
                )
            elif 'ADDITIONAL_JAVA_OPTIONS' in self._env:
                self._env['_JAVA_OPTIONS'] = self._env['ADDITIONAL_JAVA_OPTIONS']

    def _set_env(self):
        _logger.info('Preparing env.')
        self._env = os.environ
        for key, value in ast.literal_eval(self.Parameters.env).iteritems():
            self._env[key] = str(value)
        self._env['LANG'] = 'en_US.UTF-8'
        self._env['SANDBOX_TASK_LOG_DIR'] = str(self.path().joinpath('log1'))
        self._env['TEAMCITY_VERSION'] = '1'
        self._set_java_env()
        if self.Parameters.ios:
            # certs for macos clients from arcadia/sandbox/deploy/layout/sandbox_macos_mobile/usr/local/etc/openssl
            # ssl cert
            self._env['SSL_CERT_FILE'] = '/usr/local/etc/openssl/allCAs.pem'
            # cert for slack
            self._env['WEBSOCKET_CLIENT_CA_BUNDLE'] = '/usr/local/etc/openssl/DigiCertGlobalRootCA.crt'
            self._env['SSL_CERT_DIR'] = '/usr/local/etc/openssl'
        if self.Parameters.teamcity_build_id:
            self._env['TC_BUILD_ID'] = str(self.Parameters.teamcity_build_id)

        # if VCS is Arcadia (not github or bb or something else)
        # By some historic reasons, self._repo_path was incorrectly named, it has just a directory name 'arcadia'
        _logger.debug("_repo_path {}. path() {}".format(self._repo_path, self.path()))
        if self._repo_path == 'arcadia':
            # if TSR runs with export mode:
            if self._arc_exported_paths:
                _source_root_directory = self.Parameters.arc_export_prefix
            else:
                _source_root_directory = self._repo_path
            self._env['ARCADIA_ROOT'] = str(self.path().joinpath(_source_root_directory))
            self._env['ARC_WORK_TREE'] = str(self.path().joinpath(self._repo_path))
        # _logger.info(self._env)
        _logger.info('env prepared:\n{}\n'.format(self._env))

    def _update_env(self, key, value):
        if key is not None:
            self._env[key] = value
            _logger.info('added {} variable'.format(key))

    def _execute_cmd(self, args, cwd):
        self._cmd_num += 1
        _logger.info('execute cmd {}'.format(self._cmd_num))
        _logger.info('CMD: {}'.format(args))
        logger_name = 'cmd_{}'.format(self._cmd_num)
        with sdk2.ssh.Key(self, self._ssh_key.owner, self._ssh_key.name):
            rc = shellexecuter.execute_on_platform(self, args, logger_name, self.platform,
                                                       message=None, cwd=cwd)
        self.Context.cmds_info.append((args, logger_name, rc))
        return rc

    def _raise_for_status(self):
        failed = False
        failed_status = ''
        for args, logger_name, rc in self.Context.cmds_info:
            if rc:
                failed = True
                failed_status += 'Command {} died with exit code {}. See details in common.log\n'.format(args, rc)
        if failed:
            raise TaskFailure(failed_status)

    def _parse_secret_file_parameter(self, secret_src, secret_dst):
        secret_env = None
        secret_name, secret_key = secret_src.split(':')
        try:
            secret_filename, secret_env = secret_dst.split(':')
        except ValueError:
            secret_filename = secret_dst
        return os.path.join(self.work_dir, secret_filename), secret_name, secret_key, secret_env

    def _prepare_secrets_and_dependency(self):
        _logger.info('Preparing secrets, secret files and dependency.')
        secrets = ast.literal_eval(self.Parameters.secrets)
        for secret, secret_alias in secrets.iteritems():
            owner, secret_name = secret.split(':')
            self._update_env(secret_alias, sdk2.Vault.data(owner, secret_name))

        secret_files_list = ast.literal_eval(self.Parameters.secret_files)
        for secret_src, secret_dst in secret_files_list.iteritems():
            store_secret_path, secret_name, secret_key, secret_env = self._parse_secret_file_parameter(secret_src, secret_dst)
            yav_secret = yav.Secret(secret_name)
            data = base64.b64decode(yav_secret.data()[secret_key])
            with open(store_secret_path, 'w') as fh:
                fh.write(data)
            secret_abspath = os.path.abspath(store_secret_path)
            _logger.info("Wrote secret %s[%s] to %s", secret_src, secret_abspath)
            self._update_env(secret_env, secret_abspath)
            _logger.info("Set secret env variable %s = %s", secret_env, secret_abspath)

        self._dependencies = ast.literal_eval(self.Parameters.dependency_files)
        _logger.info('Secrets and dependency prepared.')

    def _disable_gradle_daemon(self):
        _logger.info('Disable gradle daemon.')
        gradle_dir = os.path.expanduser('~/.gradle')
        gradle_properties_file = os.path.join(gradle_dir, 'gradle.properties')
        if not os.path.exists(gradle_dir):
            os.makedirs(gradle_dir)
        with open(gradle_properties_file, 'wa') as f:
            f.write('org.gradle.daemon=false')
        _logger.info('Gradle daemon disabled.')

    @property
    def work_dir(self):
        if self._arc_exported_paths:
            path = str(self.path().joinpath(self.Parameters.arc_export_prefix))
        else:
            path = self._repo_path
        return os.path.join(path, self.Parameters.work_dir)

    def _execute_cmds(self):
        _logger.info('Executing cmds.')
        self.Context.cmds_info = []
        self._cmd_num = 0
        cmds = ast.literal_eval(self.Parameters.cmd)
        prepared_cmds = []
        string_preparer = StringPreparer(self._env, self._dependencies)
        for cmd in cmds:
            prepared_cmds.append(string_preparer.prepare_string(cmd))
        for cmd in prepared_cmds:
            rc = self._execute_cmd(cmd, self.work_dir)
            if rc:
                return
        _logger.info('Cmds executed.')

    def _create_dependency_files_resource(self):
        _logger.info('Creating dependency files.')
        self._dependency_files_dir = os.path.join(os.getcwd(), 'dependency_files')
        _logger.debug('_dependency_files_dir is {}'.format(self._dependency_files_dir))
        if not os.path.exists(self._dependency_files_dir):
            os.makedirs(self._dependency_files_dir)
            _logger.debug('{} did not exists. created it'.format(self._dependency_files_dir))
        with open(os.path.join(self._dependency_files_dir, 'README.md'), 'w') as f:
            f.write('Directory with internal artifacts.')

        _logger.debug("Parameters.internal_artifacts is {}".format(self.Parameters.internal_artifacts))
        for src, dst in ast.literal_eval(self.Parameters.internal_artifacts).iteritems():
            _logger.debug('Parameters.internal_artifacts src and dst are: {}, {}'.format(src, dst))
            dst = os.path.join(self._dependency_files_dir, dst)
            if not os.path.exists(os.path.dirname(dst)):
                os.makedirs(os.path.dirname(dst))
                _logger.debug('dst {} did not exists. created it')
            if os.path.isfile(src):
                shutil.copy(src, dst)
                _logger.debug("copied file from {} to {}".format(src, dst))
            elif os.path.isdir(src):
                shutil.copytree(src, dst)
                _logger.debug("copied directory {} to {}".format(src, dst))

        internal_resource = TeamcitySandboxRunnerInternalArtifacts(self,
                                                                   description='Internal artifacts',
                                                                   path=self._dependency_files_dir)
        _logger.info('Created resource TeamcitySandboxRunnerInternalArtifacts with state {}'.format(internal_resource.state))
        sdk2.ResourceData(internal_resource).ready()
        _logger.info('Resource state after executing method ready() is {}'.format(internal_resource.state))
        _logger.info('Dependency files created.')

    def on_prepare(self):
        self._ssh_key = self.Parameters.ssh_key
        self._arc_exported_paths = ast.literal_eval(self.Parameters.arc_exported_paths)
        self._repo, self._repo_path = vcs.prepare_repository(self,
                                                             self._ssh_key,
                                                             self.Parameters.repo_url,
                                                             self.Parameters.branch,
                                                             self.Parameters.commit,
                                                             True,
                                                             self.Parameters.arc_export_prefix,
                                                             self._arc_exported_paths,
                                                             )
        self.platform = shellexecuter.detect_platform(rosetta=self.Parameters.rosetta)
        _logger.debug("Set platform {}".format(self.platform))

    def _add_handlers(self):
        # logging handlers for clear info+err and err logs
        failure_fh = logging.FileHandler(self.log_path('execution_error.log').as_posix())
        failure_fh.setFormatter(logging.Formatter(LOG_FORMAT, TIME_FORMAT))
        failure_fh.setLevel(logging.ERROR)
        logging.getLogger().addHandler(failure_fh)
        success_fh = logging.FileHandler(self.log_path('execution.log').as_posix())
        success_fh.setFormatter(logging.Formatter(LOG_FORMAT, TIME_FORMAT))
        success_fh.setLevel(logging.INFO)
        logging.getLogger().addHandler(success_fh)

    def _start_artifacts_processor(self):
        artifacts_processor = ArtifactsProcessor(self, stage_task=True)
        artifacts_processor.start_processing()

    def _prepare_environments(self):
        sandbox_environments = ast.literal_eval(self.Parameters.sandbox_environments)
        if self.Parameters.ios:
            _logger.info("Started to prepare iOS environment")
            if sandbox_environments.get('xcode'):
                xcode_environment = Xcode(sandbox_environments.get('xcode'))
                xcode_environment.prepare()
                _logger.info("Prepared xcode")
            if sandbox_environments.get('jdk'):
                jdk_environment = JdkEnvironment(sandbox_environments.get('jdk'))
                jdk_environment.prepare()
                _logger.info("Prepared jdk")
            if sandbox_environments.get('certs'):
                keychain_environment = MacOsKeychainEnvironment(sandbox_environments['certs'])
                keychain_environment.prepare()
                _logger.info("Prepared keychain")
            profiles_environment = ProvisionProfileEnvironment()
            profiles_environment.prepare()
            _logger.info("Prepared Provision Profiles")
        if sandbox_environments.get('rvm+ruby'):
            rvm_ruby_environment = RvmPlusRubyEnvironment(version=sandbox_environments.get('rvm+ruby'),
                                                          platform=self.platform
                                                          )
            rvm_ruby_environment.prepare()
            _logger.info("Prepared rvm+ruby")
        if sandbox_environments.get('android-sdk'):
            _logger.info("Started to prepare Android SDK environment")
            android_sdk_environment = AndroidSdkEnvironment(sandbox_environments.get('android-sdk'))
            android_sdk_environment.prepare(self.Parameters.emulator_system_images,
                                            self.Parameters.emulator_device_skin,
                                            self.Parameters.emulator_disk_size
                                            )
            _logger.info("Prepared Android SDK environment")

    def on_execute(self):
        _logger.debug("self.Parameters: {}".format(vars(self.Parameters)))
        self._init_env()
        self._prepare_environments()
        self._set_env()
        self._prepare_secrets_and_dependency()
        self._disable_gradle_daemon()
        _logger.info("Task.id = {}. renew_caches is {}".format(sdk2.Task.current.id, self.Parameters.force_clean_build))
        caches_preparer = CachesPreparer(task=self)
        caches_preparer.prepare_caches()
        with self._repo:
            self._add_handlers()
            self._execute_cmds()
            self._start_artifacts_processor()
            self._create_dependency_files_resource()
        self._raise_for_status()
        caches_preparer.renew_caches()
