import os
import ast
import logging
import pathlib2
import shutil
import json

from sandbox.agentr.errors import ResourceNotAvailable
from sandbox.projects.common.teamcity import TeamcityArtifacts, TeamcityServiceMessagesLog
from sandbox import sdk2
from sandbox.projects.mobile_apps.teamcity_sandbox_runner.utils.html_log_reporter import HtmlLogReporter
from sandbox.projects.mobile_apps.utils.archive import Archive

_logger = logging.getLogger('artifacts_processor')


class Artifact(dict):

    """
    Example:
        Input:
            artifact: ('+output_log': 'samples.zip')

        Parsed result:
            src: output_log
            full_src: $artifacts_resource_path/output_log
            dst_basename_noext: samples (STAGE task will create folder 'samples' and copy output_log
                                to $artifacts_resource_path/samples

            full_dst: samples.zip
            sign: +
    """

    def __init__(self, artifact, artifacts_resource_path="", teamcity_artifacts_dir="", string_preparer=None):
        _logger.debug("Artifact. Parsing artifact: {}".format(artifact))
        self.string_preparer = string_preparer
        self.artifacts_resource_path = artifacts_resource_path
        self.teamcity_artifacts_dir = teamcity_artifacts_dir
        self.parse_artifact(artifact)
        dict.__init__(self)

    def _expand_str(self, line):
        if self.string_preparer:
            line = self.string_preparer.prepare_string(line)
        return line

    def parse_artifact(self, artifact):
        # artifact: (src: string, dst: string)
        current_folder = './'
        known_operations = '+-$'
        known_archive_types = ('zip', 'tgz')

        _logger.debug("parse_artifact. artifact: {}".format(artifact))
        if len(artifact) == 2:
            _src, _dst = artifact
        else:
            _src = artifact
            _dst = None

        if _src[0] in known_operations:
            self.sign, _src = _src[0], _src[1:]
        else:
            self.sign = '+'

        # remove leading './'
        if _src.startswith(current_folder):
            _src = _src[len(current_folder):]

        self.src_archived = _src.endswith(known_archive_types)

        self.src = self._expand_str(_src)
        self.full_src = os.path.join(self.artifacts_resource_path, self.src)

        self.prefix = os.path.dirname(self.full_src[:self.full_src.find("*")])

        self.dst_archived = False
        if _dst:
            _dst = self._expand_str(_dst)
            self.full_dst = os.path.join(self.teamcity_artifacts_dir, _dst)
            if _dst.endswith('.tgz'):
                self.compression = 'gz'
            else:
                self.compression = None
            if _dst.endswith(known_archive_types) and not self.src_archived:
                self.dst_archived = True
                _dst = _dst[:-4]
        self.dst_basename_noext = _dst

        _logger.debug("parse_artifact result: sign: {}, src: {}, "
                      "src prefix: {}, full_src: {}, src archived: {}, "
                      "dst_basename_noext: {}, dst archived: {}".format(self.sign,
                                                                        self.src,
                                                                        self.prefix,
                                                                        self.full_src,
                                                                        self.src_archived,
                                                                        self.dst_basename_noext,
                                                                        self.dst_archived))


class ArtifactsProcessor(object):
    def __init__(self, task, stage_task=None, string_preparer=None):
        self._task = task
        self._stage_task = stage_task
        self._string_preparer = string_preparer

    def _generate_html_report(self):
        _logger.info('Try to create reports files.')

        reports_map = {
            self._html_reporter.text_log: 'full_log.txt',
            self._html_reporter.short_html: 'index.html',
            self._html_reporter.html: 'full_index.html',
        }

        for report, filename in reports_map.iteritems():
            full_path = os.path.join(self._teamcity_artifacts_dir, filename)
            with open(full_path, 'w') as fh:
                fh.write(report)

        with open(self._teamcity_messages_path, 'a') as f:
            f.write(self._html_reporter.teamcity_messages)
        _logger.info('Reports file successfully created.')

    def _clean_logs(self, log_dir):
        _logger.info('Cleaning junk files.')
        junk_files = ['disk_usage.yaml', 'peak_disk_usage.yaml']
        for junk_file in junk_files:
            junk_file = os.path.join(log_dir, junk_file)
            if os.path.exists(junk_file):
                os.remove(junk_file)
        _logger.info('Junk files cleaned.')

    @staticmethod
    def _read_file(dir_path, filename):
        file_path = os.path.join(dir_path, filename)
        data = ''
        if os.path.exists(file_path):
            with open(file_path, 'r') as fh:
                data = fh.read().strip()
        return data

    def _copy_logs(self, task_id, name):
        _logger.info('Try to copy child TASK_LOGS from {} {}.'.format(task_id, name))
        log_resource = sdk2.Resource.find(task_id=task_id, type='TASK_LOGS', state='READY').first()
        if log_resource:
            _logger.info('The TASK_LOGS resource was found and started to be processed.')
            log_dir_path = '{}/{}_logs'.format(self._logs_dir, name)
            try:
                resource_log_path = self._retriable_get_resource_path(log_resource)
                shutil.copytree(resource_log_path, log_dir_path)
                _logger.info('Logs copied.')
                if task_id != self._task.id:
                    info_log = self._read_file(log_dir_path, 'execution.log')
                    err_log = self._read_file(log_dir_path, 'execution_error.log')
                    self._html_reporter.append_stage_log(info_log, err_log, name, int(task_id in self._task.Context.failed_tasks))
                _logger.info('TASK_LOGS processed.')
            except ResourceNotAvailable:
                _logger.info("Failed to process TASK_LOGS: resource is not available")
        else:
            _logger.info('TASK_LOGS resource is not found.')

    def _pack_archives(self):
        _logger.info('Try to pack archives {}.'.format(str(self._task.Context.archives)))
        for item in self._task.Context.archives:
            if os.path.exists(item.dst_basename_noext):
                Archive.pack(archive=os.path.basename(item.full_dst),
                             archive_folder=os.path.dirname(item.full_dst),
                             src=item.dst_basename_noext,
                             compression=item.compression)
                os.chmod(item.dst_basename_noext, 0o777)
                shutil.rmtree(item.dst_basename_noext)
            else:
                _logger.warning('Failed to create an archive because the directory "{}" does not exist.'.format(item.dst_basename_noext))
        _logger.info('Archives packing finished.')

    def _copy(self, src, dst):
        _logger.info('Copying from src: "{}"\t dst: "{}"'.format(src, dst))
        try:
            if os.path.isfile(src):
                file_dir = os.path.dirname(dst)
                if file_dir and not os.path.exists(file_dir):
                    os.makedirs(file_dir)
                shutil.copy(src, dst)
            elif os.path.isdir(src):
                shutil.copytree(src, dst)
        except (OSError, IOError) as e:
            _logger.exception(e)
            self._task.Context._failed_messages.append('An error occurred while working with path {}. Check that the paths of the artifacts do not match.'.format(src))

    def _retriable_get_resource_path(self, resource):
        from library.python.retry import retry_call, RetryConf
        retryconf = RetryConf(logger=_logger).waiting(delay=2., backoff=2., jitter=1.).upto(minutes=5.).on(
            ResourceNotAvailable)

        resource_path = retry_call(
            self._get_resource_path,
            f_kwargs={'resource': resource},
            conf=retryconf)

        return resource_path

    def _get_resource_path(self, resource, *args):
        _logger.debug("_get_resource params: {}".format(resource))
        return sdk2.ResourceData(resource).path.as_posix()

    def _copy_artifacts(self, task_id, stage_name):
        _logger.info('Try to copy child TEAMCITY_ARTIFACTS from {} {}.'.format(task_id, stage_name))
        artifacts_resource = sdk2.Resource.find(task_id=task_id, type='TEAMCITY_ARTIFACTS', state='READY').first()
        if artifacts_resource:
            _logger.info('The TEAMCITY_ARTIFACTS resource was found. {}'.format(artifacts_resource))
            artifacts_resource_path = self._retriable_get_resource_path(artifacts_resource)
            _logger.info('And started to be processed.')
            for _artifact in self._task.Context.artifacts[stage_name].items():
                artifact = Artifact(_artifact, artifacts_resource_path, self._teamcity_artifacts_dir, self._string_preparer)
                if artifact.sign == '-':
                    continue
                _logger.info('Started to process artifact from "{}" to "{}".'.format(artifact.src, artifact.dst_basename_noext))
                if artifact.dst_archived:
                    _logger.info('{} will be archived.'.format(artifact.dst_basename_noext))
                    # self._task.Context.archives.append(artifact.full_dst)
                    self._task.Context.archives.append(artifact)

                if artifact.src.count('*'):
                    _logger.info('Src path "{}" will be processed with using glob.'.format(artifact.src))
                    for cur_src in pathlib2.Path(artifacts_resource_path).glob(artifact.src):
                        cur_src = os.path.join(artifacts_resource_path, cur_src.as_posix())
                        full_dst = os.path.join(self._teamcity_artifacts_dir, artifact.dst_basename_noext, cur_src[len(artifact.prefix) + 1:])
                        self._copy(cur_src, full_dst)
                else:
                    _logger.info('Src path "{}" will be processed as complete path.'.format(artifact.src))
                    if not os.path.exists(artifact.full_src):
                        error_msg = '"{}" artifact path does not exists.'.format(artifact.full_src)
                        _logger.error(error_msg)
                        self._task.Context._failed_messages.append(error_msg)
                        continue
                    self._copy(artifact.full_src, artifact.dst_basename_noext)
            _logger.info('TEAMCITY_ARTIFACTS processed.')
        else:
            _logger.info('TEAMCITY_ARTIFACTS resource is not found.')

    def _move(self, src):
        real_destination = os.path.join(self._teamcity_artifacts_dir, src)
        _logger.info('Moving {} to {}.'.format(src, real_destination))
        if os.path.exists(real_destination):
            return
        if os.path.dirname(real_destination) != real_destination[:-1] and not os.path.exists(os.path.dirname(real_destination)):
            os.makedirs(os.path.dirname(real_destination))
        if os.path.isfile(src):
            shutil.copy(src, real_destination)
        elif os.path.isdir(src):
            shutil.copytree(src, real_destination)
        _logger.info('Moved {} to {}.'.format(src, real_destination))

    def _remove(self, src):
        _logger.info('Removing {}'.format(src))
        if os.path.isfile(src):
            os.remove(src)
        elif os.path.isdir(src):
            shutil.rmtree(src)
        _logger.info('{} removed'.format(src))

    def _file_processing(self, files, process_func):
        _logger.info('Processing artifacts.')
        for file in files:
            _logger.info('Processing file {}.'.format(file))
            try:
                match_files = list(pathlib2.Path('.').glob(file))
                if len(match_files) == 0:
                    _logger.warning('File "{}" not found.'.format(file))
                _logger.info(match_files)
                for match_file in match_files:
                    process_func(str(match_file))
            except NotImplementedError:
                error_msg = "File pattern {} must be relative. Stopped processing artifact. See https://docs.python.org/dev/library/pathlib.html#pathlib.Path.glob".format(
                    files)
                _logger.error(error_msg)
        _logger.info('Artifacts processed.')

    def _list_work_dir(self, work_dir):
        if self._task.Parameters.release_type != 'test':
            return
        _logger.info('Listing "{}"'.format(work_dir))
        if self._task.Parameters.work_dir in ['.', './', '']:
            _logger.info('Skipped.')
            return
        tree_str = ''
        for root, dirs, files in os.walk(work_dir):
            level = root.replace(work_dir, '').count(os.sep)
            indent = ' ' * 4 * (level)
            tree_str = '{}{}{}/\n'.format(tree_str, indent, os.path.basename(root))
            subindent = ' ' * 4 * (level + 1)
            for f in files:
                tree_str = '{}{}{}\n'.format(tree_str, subindent, f)
        with open(self._task.log_path('work_dir_listing.txt').as_posix(), 'w') as f:
            f.write(tree_str)
        _logger.info('Listed.')

    def _process_stage_artifacts(self):
        self._list_work_dir(self._task.work_dir)
        removed = []
        published = []
        for cur_path in ast.literal_eval(self._task.Parameters.artifacts):
            artifact = Artifact(cur_path)
            if artifact.sign == '-':
                removed.append(artifact.src)
            elif artifact.sign == '+':
                published.append(artifact.src)

        self._file_processing(removed, self._remove)
        self._file_processing(published, self._move)

    def _prepare_teamcity_artifacts_dir(self):
        _logger.info('Try to prepare teamcity resource.')
        if self._stage_task or not self._task.Parameters.use_parent_resources:
            self._teamcity_artifacts_dir = os.path.join(os.getcwd(), 'teamcity_artifacts')
            if not os.path.exists(self._teamcity_artifacts_dir):
                os.makedirs(self._teamcity_artifacts_dir)
            if not self._stage_task:
                self._teamcity_messages_path = str(self._task.path('teamcity_messages.log'))
        else:
            self._teamcity_artifacts_dir = os.path.join(os.getcwd(), 'parent_teamcity_artifacts')
            self._parent_teamcity_artifacts_dir = sdk2.ResourceData(self._task.Parameters.teamcity_artifacts_resource).path.as_posix()
            self._teamcity_messages_path = sdk2.ResourceData(self._task.Parameters.teamcity_messages_resource).path.as_posix()
        if self._stage_task:
            with open(os.path.join(self._teamcity_artifacts_dir, 'artifacts_info'), 'w') as f:
                f.write('Artefacts prepared by {} task.'.format(self._task.id))
        _logger.info('Teamcity resource prepared.')

    def _aggregate_artifacts(self):
        _logger.info('Try to aggregate artifacts.')
        if not os.path.exists(self._teamcity_artifacts_dir):
            os.makedirs(self._teamcity_artifacts_dir)
        cwd = os.getcwd()
        os.chdir(self._teamcity_artifacts_dir)
        with open(os.path.join(self._teamcity_artifacts_dir, 'config.json'), 'w') as config_file:
            json.dump(self._task.Context.config, config_file)
        self._logs_dir = self._teamcity_artifacts_dir + '/logs'
        if not os.path.exists(self._logs_dir):
            os.makedirs(self._logs_dir)
        self._copy_logs(self._task.id, 'teamcity_runner')
        for stage_name, task_id in self._task.Context.sub_tasks_ids.iteritems():
            self._copy_logs(task_id, stage_name)
            self._copy_artifacts(task_id, stage_name)
        self._pack_archives()
        os.chdir(cwd)

        with open(self._teamcity_messages_path, 'w') as fout:
            build_number_message = ''
            if self._task.Context.build_number:
                build_number_message = '\n##teamcity[buildNumber \'{}\']'.format(self._task.Context.build_number)
            fout.write('##teamcity[publishArtifacts \'teamcity_artifacts => .\']\n##teamcity[importData type=\'junit\' path=\'teamcity_artifacts/junit/**/*.xml\']{}'.format(build_number_message))
        _logger.info('Artifacts aggregated.')

    def _ready_resources(self):
        _logger.info('Try to ready resources.')
        if self._stage_task or not self._task.Parameters.use_parent_resources:
            # you should create messages first because of the publishing specifics
            if not self._stage_task:
                sdk2.ResourceData(TeamcityServiceMessagesLog(self._task, 'TeamCity messages', self._teamcity_messages_path, auto_backup=True)).ready()
            sdk2.ResourceData(TeamcityArtifacts(self._task, 'TeamCity artifacts', self._teamcity_artifacts_dir, auto_backup=True)).ready()
        else:
            shutil.move(self._teamcity_artifacts_dir, self._parent_teamcity_artifacts_dir)
            sdk2.ResourceData(self._task.Parameters.teamcity_messages_resource).ready()
            sdk2.ResourceData(self._task.Parameters.teamcity_artifacts_resource).ready()
        _logger.info('Resources ready.')

    def start_processing(self):
        self._task.Context.archives = []
        self._prepare_teamcity_artifacts_dir()
        if self._stage_task:
            self._process_stage_artifacts()
        else:
            self._html_reporter = HtmlLogReporter()
            self._aggregate_artifacts()
            self._generate_html_report()
        self._ready_resources()
        self._task.Context.archives = []
