import ast
import logging
import os
import random
import shutil
import string

from datetime import datetime

from sandbox import sdk2

from sandbox.projects.mobile_apps.utils.archive import Archive
from sandbox.projects.mobile_apps.teamcity_sandbox_runner.utils.resources import TeamcitySandboxRunnerCache

logger = logging.getLogger('caches_preparer')


class CachesPreparer(object):

    def __init__(self, task):
        """
        self.caches is dict
            example:
            {<cache_path>: {use_config_hash: False, resource_id: 0, shared_common_name: "common_name"},
             ..
            }
        """
        self.task = task
        self.task_name = self.task.Parameters.name
        self.task_id = self.task.id
        self.caches = self._parse_caches_parameter(ast.literal_eval(self.task.Parameters.caches))
        self.config_hash = self.task.Parameters.config_hash
        self.force_clean_build = self.task.Parameters.force_clean_build
        self.platform = self.task.platform

    @staticmethod
    def _parse_caches_parameter(lines):
        """
        lines example:
            ~/.gradle/caches/modules-2   # stage specific cache
            ~/.cocoapods:use_config_hash
            ~/.gradle/wrapper:shared:<shared_common_name>    # cache shared via name 'shared_common_name'
            ~/.test:<resource_id> # cache pinned with resource_id
        """
        parsed_caches = {}
        for line in lines:
            cache_path, _, option = line.partition(":")
            cache_parameters = {"use_config_hash": False, "resource_id": 0, "shared_common_name": ""}
            if option == "use_config_hash":
                cache_parameters["use_config_hash"] = True
            elif str(option).isdigit():
                cache_parameters["resource_id"] = int(option)
            elif str(option).startswith("shared:"):
                _, _, cache_parameters["shared_common_name"] = str(option).partition(":")
            parsed_caches[cache_path] = cache_parameters
        return parsed_caches

    def _make_attributes(self, cache, cache_params):
        """
        attributes priorities, top down:
            - config_hash
            - shared_common_name (with this option, cache_owner value is ignored
        """
        attrs = {'target': cache, 'platform': self.platform, 'cache_owner': self.task_name}
        if cache_params["use_config_hash"]:
            attrs["config_hash"] = self.config_hash
        elif cache_params["shared_common_name"]:
            attrs["shared_common_name"] = cache_params["shared_common_name"]
            attrs.pop('cache_owner')
        logger.debug("{} attributes: {}".format(cache, attrs))
        return attrs

    def _get_cache_resource(self, cache, cache_params):
        """
        :param cache: List element where List is teamcity_sandbox_runner_stage parameter caches.
                        Value examples: "~/.gradle/wrapper", "~/.cocoapods"
        :param cache_params:
                        {use_config_hash: False, resource_id: 0, shared_common_name: "common_name"}
        :return: sdk2.Resource or None, is_resource_usage_forced: boolean
        """
        if cache_params["resource_id"] != 0:
            resource = sdk2.Resource[cache_params["resource_id"]]
            is_resource_usage_forced = True
        else:
            attrs = self._make_attributes(cache, cache_params)
            resource = sdk2.Resource.find(type='TEAMCITY_SANDBOX_RUNNER_CACHE', attrs=attrs).first()
            is_resource_usage_forced = False
        logger.info("Resource found: {}".format(resource))
        return resource, is_resource_usage_forced

    def prepare_caches(self):
        for cache_path, cache_params in self.caches.items():
            logger.info('Start {} preparing.'.format(cache_path))

            if not cache_path.startswith('~/'):
                logger.debug('Ignoring cache {}. Reason: it does not start with ~'.format(cache_path))
                continue

            resource, is_resource_usage_forced = self._get_cache_resource(cache_path, cache_params)

            if not resource or resource.state != 'READY':
                logger.info("Cache resource not found or corrupted. Skip unpacking")
                continue

            if not is_resource_usage_forced and is_cache_generation_needed(resource, self.force_clean_build):
                logger.info("Skipped cache_resource {} unpacking".format(cache_path))
                continue

            clean_directory(cache_path)
            unpack_cache(resource, cache_path)
            logger.info('Unpacked resource to {}'.format(cache_path))

    def _add_note(self, directory, cache):
        """
        python zipfile can't store empty directory in zip archive: archive is created, but directory is missed.
        After "unpacking" empty archive, cache folder doesn't exist and if stage expects it, it will fail.
        :param directory:
        :param cache:
        :return:
        """
        with open(os.path.join(directory, 'info.txt'), 'w') as f:
            f.write('cache with {} from {} task'.format(cache, self.task_id))

    def _generate_cache_resource(self, cache, cache_params, cache_dir):
        logger.info('Generating cache resource for {}.'.format(cache))
        archive_filename = '{}.zip'.format(cache.replace('~/', 't_').replace('/', '_'))
        expanded_cache_path = os.path.expanduser(cache)
        if not os.path.exists(expanded_cache_path):
            logger.info('{} is empty. Skip cache {} regeneration'.format(expanded_cache_path, cache))
            return

        self._add_note(expanded_cache_path, cache)
        Archive.pack(archive=archive_filename, archive_folder=cache_dir, src=expanded_cache_path)

        attrs = self._make_attributes(cache, cache_params)
        cache_resource = TeamcitySandboxRunnerCache(self.task, '{} cache'.format(cache),
                                                    os.path.join(cache_dir, archive_filename), **attrs)
        sdk2.ResourceData(cache_resource)
        logger.info('New resource for {} was generated: {}'.format(cache, cache_resource.id))

    def renew_caches(self):
        tmp_cache_dir = create_random_directory(os.getcwd())
        for cache, cache_params in self.caches.items():
            logger.info('Renew {} cache'.format(cache))
            resource, _ = self._get_cache_resource(cache, cache_params)
            if not resource or resource.state != 'READY' or is_cache_generation_needed(resource, self.force_clean_build):
                self._generate_cache_resource(cache, cache_params, tmp_cache_dir)
            else:
                logger.info("No need to regenerate cache {}".format(cache))


def clean_directory(cache_path):
    logger.info('Try to clean cache destination directory')
    src = os.path.expanduser(cache_path)
    if os.path.exists(src):
        shutil.rmtree(src)
        logger.info('{} has been erased.'.format(cache_path))
    else:
        logger.info("Nothing to clean - cache destination directory doesn't exist.")


def unpack_cache(resource, cache_dst_path):
    logger.info('Unpack CACHE_RESOURCE: {}.'.format(resource.id))
    cache_path = sdk2.ResourceData(resource).path.as_posix()
    Archive.unpack(cache_path, os.path.expanduser(cache_dst_path))
    logger.info('Resource {} has been unpacked to {}.'.format(resource.id, cache_dst_path))


def is_cache_generation_needed(resource, force_clean_build):
    """
    :param resource: result of sdk2.Resource.find() call. resource must exist and has state READY
    :param force_clean_build: boolean
    :return: True if SB resource is older than 3 days OR cache generation is forced
    """
    if force_clean_build:
        logger.info('Cache resource rebuilding is forced')
        return True
    if calculate_cache_age_in_days(resource) > 3:
        logger.info('Cache was created more than 3 days ago. Cache resource rebuilding is needed')
        return True
    return False


def calculate_cache_age_in_days(resource):
    return (datetime.now() - resource.created.replace(tzinfo=None)).days


def create_random_directory(directory):
    """
    I've tried mkdtemp(dir=os.getcwd()) and SB task failed with an error
    that <task_path>/<mktemp directory>/<archive> doesn't exist.
    However os.stat() had shown that directory exist - https://paste.yandex-team.ru/9342443
    And then this function was implemented.
    """
    letters = string.ascii_lowercase
    random_local_dir = ''.join(random.choice(letters) for _ in range(10))
    full_path = os.path.join(directory, random_local_dir)
    os.mkdir(full_path)
    return full_path
