import logging
import os
import re

from sandbox.common.errors import TaskFailure
from sandbox import sdk2

_logger = logging.getLogger('string_preparer')


class StringPreparer(object):

    def __init__(self, env, dependencies=None):
        self._env = env
        self._dependencies = dependencies if dependencies else {}

    def prepare_string(self, input_string):
        _logger.info('Preparing string {}.'.format(input_string))
        founded_envs = 1
        original_string = input_string
        while founded_envs > 0:
            original_string = input_string
            params = re.findall('%.*?%', input_string)
            _logger.info('{} found in string: {}'.format(params, original_string))
            founded_envs = len(params)
            for i in range(len(params)):
                param = params[i]
                if param.startswith('%env.'):
                    replaced_str = self._process_env_param(param)
                elif param.startswith('%internal.'):
                    if self._dependencies is None:
                        raise TaskFailure('Could not use \"internal.<>\" here: \"{}\"'.format(param))
                    replaced_str = self._process_internal_param(param)
                else:
                    raise TaskFailure('Parameter \"{}\" not found in known params'.format(param))

                input_string = input_string.replace(param, replaced_str)

        self._check_unknown_params(original_string, input_string)
        _logger.info('String prepared.')
        _logger.info('Result String: {}'.format(input_string))
        return input_string

    def _process_env_param(self, full_param):
        parsed_param = full_param[5:-1]
        if parsed_param not in self._env:
            raise TaskFailure('Parameter \"{}\" not found in extra_params'.format(full_param))
        replaced_str = self._env[parsed_param]
        return replaced_str

    def _process_internal_param(self, full_param):
        parsed_param = full_param[10:-1]
        task_name, path = parsed_param.split(':')
        internal_artifacts_resource = sdk2.Resource.find(task_id=self._dependencies[task_name],
                                                         type='TEAMCITY_SANDBOX_RUNNER_INTERNAL_ARTIFACTS').first()
        replaced_str = os.path.join(sdk2.ResourceData(internal_artifacts_resource).path.as_posix(), path)
        if not os.path.exists(replaced_str):
            raise TaskFailure("Path \"{}\" not exists".format(replaced_str))
        return replaced_str

    def _check_unknown_params(self, original_string, result_string):
        unknown_params = re.findall('%.*?%', result_string)
        if len(unknown_params):
            raise TaskFailure('Unknown params {} in CMD: {}'.format(unknown_params, original_string))
