import hashlib
import logging
import os
import yaml

from sandbox import sdk2
from sandbox.sdk2.vcs.git import Git
from sandbox.common.errors import TaskFailure, SubprocessError
from sandbox.projects.common.vcs.arc import Arc

_logger = logging.getLogger('utils')

SCHEMA = {
    "type": "object",
    "properties": {
        "template": {
            "type": "array",
            "items": {
                "type": "string"
            }
        },
        "config": {
            "type": "object",
            "properties": {
                "name": {"type": "string"},
                "build_counter": {"type": "string"},
                "runner_version": {"type": "string"}
            },
            "required": ["name", "runner_version"]
        }
    },
    "required": ["config", "stages"]
}

STAGE_SCHEMA = {
    "type": "object",
    "properties": {
        "work_dir": {"type": "string"},
        "cmd": {
            "type": "array",
            "items": {
                "type": "string"
            }
        },
        "fail_fast": {"type": "boolean"},
        "lxc": {"type": "number"},
        "env": {"type": "object"},
        "secrets": {"type": "object"},
        "secret_files": {"type": "object"},
        "kill_timeout": {"type": "number"},
        "artifacts": {
            "type": "object",
            "additionalProperties": {"type": "string"}
        },
        "caches": {
            "type": "array",
            "items": {
                "type": "string"
            }
        },
        "junit": {
            "type": "array",
            "items": {
                "type": "string"
            }
        },
        "depends_on": {
            "type": "array",
            "items": {
                "type": "string"
            }
        },
        "internal_artifacts": {"type": "object"},
        "multislot": {"type": "string"},
        "ios": {"type": "boolean"},
        "rosetta": {"type": "boolean"},
        "xcode": {"type": "string"},
        "certs": {
            "type": "array",
            "items": {
                "type": "string"
            }
        },
        "rvm+ruby": {"type": "string"},
        "jdk": {"type": "string"},
        "disk_space": {"type": "integer"},
        "emulator_system_images": {"type": "string"},
        "ramdisk": {"type": "integer"},
        "arc_exported_paths": {
            "type": "array",
            "items": {
                "type": "string"
            }
        },
    },
    "required": ["cmd"]
}


class GitMock(object):
    """ Mock Git context manager. """

    def __init__(self):
        pass

    def __enter__(self):
        pass

    def __exit__(self, exc_type, exc_val, exc_tb):
        pass


def get_config(config_path):
    try:
        with open(config_path, 'r') as f:
            return f.read()
    except Exception as e:
        raise TaskFailure(e)


def read_config(config_from_repository, config_path, config_str, config_for_launcher, dependent_templates):
    """
    :returns: config, sha1 config hash
    """
    from jsonschema import validate

    _logger.info(dependent_templates)
    config = {}
    if config_from_repository:
        _logger.info('Reading config from repository.')
        path_template = '{}/{}'
        if config_path.endswith('.'):
            path_template = '{}{}'
        config_str = get_config(path_template.format(config_path, 'config.yaml'))
    else:
        _logger.info('Reading config from Parameter.')
    if config_for_launcher:
        stages_pos = config_str.find('stages')
        config_str = config_str[:stages_pos] if stages_pos != -1 else config_str
    else:
        for dependent_template in dependent_templates[::-1]:
            config_str = '{}\n{}'.format(get_config(dependent_template), config_str)
    config = yaml.safe_load(config_str)
    if not config_for_launcher:
        validate(config, SCHEMA)
        for _, stage in config['stages'].items():
            validate(stage, STAGE_SCHEMA)
    config_hash = hashlib.sha1(str(config).encode('utf-8')).hexdigest()
    _logger.debug('config: {}'.format(config))
    _logger.debug('config hash: {}'.format(config_hash))
    _logger.info('Config read.')
    return config, config_hash


def _clone_repository(task, ssh_key, repo_url, branch, commit=''):
    _logger.info('Cloning repository.')
    repo_obj = None
    repo_point = ''
    if (repo_url.find('git@') != -1):
        try:
            git = Git(repo_url, filter_branches=False)
            with sdk2.ssh.Key(task, ssh_key.owner, ssh_key.name):
                git.clone('.', branch, commit=commit)
            repo_obj = GitMock()
        except SubprocessError as e:
            _logger.info("Git clone exception: {}".format(e))
            raise TaskFailure("Git clone failed")
    else:
        arc = Arc()
        with sdk2.ssh.Key(task, ssh_key.owner, ssh_key.name):
            repo_point = 'arcadia'
            repo_path = ''
            fetch_flag = False
            _logger.debug(
                "arc mount parameters: path={} changeset={}, mount_point={}, fetch_all={}".format(
                    repo_path,
                    commit,
                    repo_point,
                    fetch_flag,
                )
            )
            repo_obj = arc.mount_path(path=repo_path,
                                      changeset=commit,
                                      mount_point=repo_point,
                                      fetch_all=fetch_flag,
                                      )
    _logger.info('Repository cloned.')
    return repo_obj, repo_point


def _arc_export_directories(task, ssh_key, arc_export_prefix, arc_exported_paths, commit='HEAD'):
    _logger.info("Exporting arc directories")
    _logger.debug("arc_exported_paths: {}".format(arc_exported_paths))

    saved_current_path = os.getcwd()
    mount_point = str(task.path().joinpath('arcadia_bare'))
    arcadia_export_dir = str(task.path().joinpath(arc_export_prefix))

    os.mkdir(arcadia_export_dir)
    os.mkdir(mount_point)
    os.chdir(mount_point)

    _logger.debug("Current dir {}".format(os.getcwd()))
    arc = Arc()
    _logger.debug("init bare arc")
    with arc.init_bare(mount_point):
        for repo_path in arc_exported_paths:
            _logger.debug("Arc export {}, {}, {} to {}".format(mount_point, commit, repo_path, arcadia_export_dir))
            result = arc.export(mount_point, commit, repo_path, arcadia_export_dir)
            _logger.debug("Export result: {}".format(result))
            _logger.debug("arcadia_export_dir content: {}".format(os.listdir(arcadia_export_dir)))
    os.chdir(saved_current_path)
    _logger.info("Exported directories")


def prepare_repository(task, ssh_key, repo_url, branch, commit, config_from_repository, arc_export_prefix=None, arc_exported_paths=None):
    _logger.info('Preparing repository.')
    repo = GitMock()
    repo_path = ''
    if config_from_repository:
        from library.python.retry import retry_call, RetryConf
        from sandbox.sdk2.helpers.process import subprocess
        retryconf = RetryConf(logger=_logger).waiting(delay=2., backoff=2., jitter=1.).upto(minutes=5.).on(
            TaskFailure, subprocess.SubprocessError, subprocess.TimeoutExpired, Exception)
        repo, repo_path = retry_call(_clone_repository, (task, ssh_key, repo_url, branch, commit), conf=retryconf)
        if arc_exported_paths:
            _arc_export_directories(task, ssh_key, arc_export_prefix, arc_exported_paths, commit)

    _logger.info('Repository prepared.')
    _logger.debug('prepare_repository returns repo {}, repo_path {}'.format(repo, repo_path))
    return repo, repo_path


def prepare_repository_and_read_config(task, ssh_key, repo_url, branch, commit, config_from_repository, config_path,
                                       config_str, config_for_launcher, dependent_templates=[]):
    repo, repo_path = prepare_repository(task, ssh_key, repo_url, branch, commit, config_from_repository)
    with repo:
        dependent_templates = list([os.path.join(repo_path, x) for x in dependent_templates])
        config, config_hash = read_config(config_from_repository, os.path.join(repo_path, config_path), config_str,
                                          config_for_launcher, dependent_templates)
        return config, config_hash, repo_path
