import os
import logging
import time
from sandbox.sandboxsdk.environments import SandboxEnvironment
from sandbox.projects.mobile_apps.utils.resources import AndroidSdk
from sandbox.projects.mobile_apps.utils import shellexecuter as ut

_logger = logging.getLogger(__name__)


class AndroidSdkEnvironment(SandboxEnvironment):
    """ Sandbox environment to work with Android SDK. """
    resource_type = AndroidSdk

    """ Sandbox environment to work with Android Emulator. """
    LD_LIBRARY_PATH = "$ANDROID_HOME/emulator/lib64/:$ANDROID_HOME/emulator/lib64/gles_mesa:$ANDROID_HOME/emulator/lib64/gles_swiftshader:$ANDROID_HOME/emulator/lib64/qt/lib/:$LD_LIBRARY_PATH"
    LD_PREPARE = 'ldconfig -C /tmp/ld_cache'
    CREATE_EMULATOR = 'echo "n" | $ANDROID_HOME/cmdline-tools/tools/bin/avdmanager create avd -n TestEmulator -k "system-images;{}" -d "{}"'
    RUN_EMULATOR = '$ANDROID_HOME/emulator/emulator -no-snapshot -no-boot-anim ' \
                   '-accel on -no-window -no-audio -partition-size {} -gpu swiftshader_indirect ' \
                   '-verbose -avd TestEmulator -verbose -qemu -enable-kvm'
    TURNOFF_ANIMATION = '$ANDROID_HOME/platform-tools/adb shell settings put global window_animation_scale 0; ' \
        '$ANDROID_HOME/platform-tools/adb shell settings put global transition_animation_scale 0; ' \
        '$ANDROID_HOME/platform-tools/adb shell settings put global animator_duration_scale 0'

    def __init__(self, version=None, **kws):
        super(AndroidSdkEnvironment, self).__init__(version, 'linux', **kws)

    def prepare_emulator(self, system_images, device_skin="Nexus 5", disk_size=1):
        self.update_os_env(value='LD_LIBRARY_PATH', key=self.LD_LIBRARY_PATH)
        ut.run(self.LD_PREPARE, timeout=60)
        ut.execute(self.CREATE_EMULATOR.format(system_images, device_skin), in_background=False, timeout=60)
        ut.execute(self.RUN_EMULATOR.format(disk_size * 1024), in_background=True)
        time.sleep(120)
        ut.execute('$ANDROID_HOME/platform-tools/adb wait-for-device', in_background=False, timeout=60)
        ut.execute(self.TURNOFF_ANIMATION, in_background=False, timeout=60)

    def prepare(self, emulator_system_images, device_skin, disk_size):
        resource_path = self.get_environment_resource()
        self.extract_tar(resource_path, os.path.expanduser('~'))
        android_sdk_path = "{}/android-sdk".format(os.path.expanduser('~'))
        self.update_os_env('ANDROID_HOME', android_sdk_path)
        self.update_os_env('ANDROID_SDK_ROOT', android_sdk_path)
        if emulator_system_images:
            self.prepare_emulator(emulator_system_images, device_skin, disk_size)
        self.check_environment()
        return android_sdk_path
