import logging
import os
import shutil
import tarfile
from zipfile import ZipFile, ZipInfo, ZIP_DEFLATED


logger = logging.getLogger('Archive')


class PatchedZipFile(ZipFile):

    def extract(self, member, path=None, pwd=None):
        if not isinstance(member, ZipInfo):
            member = self.getinfo(member)

        if path is None:
            path = os.getcwd()

        ret_val = self._extract_member(member, path, pwd)
        attr = member.external_attr >> 16
        os.chmod(ret_val, attr)
        return ret_val


class ArchiveTar(object):

    @staticmethod
    def pack(archive, archive_folder, src, compression=None):
        """
        src: folder, which should be archived
        archive: archive filename
        archive_folder: archive's destination folder
        """
        logger.info('Tar {} to {}/{}'.format(src, archive_folder, archive))
        compression = '' if compression is None else compression
        mode = 'w|{}'.format(compression)
        dst_path = os.path.join(archive_folder, archive)
        try:
            with tarfile.open(dst_path, mode) as arc_fh:
                arc_fh.add(src, arcname='.')
        except Exception as e:
            logger.exception(e)
        logger.info("Tarred {} to {}".format(src, dst_path))

    @staticmethod
    def unpack(archive, folder, compression=None):
        logger.info("Untar {} to {}".format(archive, folder))
        compression = '' if compression is None else compression
        mode = 'r|{}'.format(compression)
        try:
            with tarfile.open(archive, mode) as arc_fh:
                arc_fh.extractall(folder)
        except Exception as e:
            logger.exception(e)
        logger.info("Untarred {}".format(archive))
        logger.debug('Folder {} has unpacked content: {}'.format(folder, os.listdir(folder)))


class ArchiveZip(object):
    @staticmethod
    def pack(archive, archive_folder, src, compression=None):
        """
        src: folder, which should be zipped
        archive: archive filename like a.zip
        archive_folder: folder where to place archive

        This function has weird chmod 777 for src folder before deletion and at adding files and directories to archive
         implemented in https://st.yandex-team.ru/DEVTOOLS-6387.
        """
        logger.info('Zipping {} to {}/{}'.format(src, archive_folder, archive))
        # archive = '{}.zip'.format(archive)
        with ZipFile(archive, 'w', compression=ZIP_DEFLATED, allowZip64=True) as arc_fh:
            cwd = os.getcwd()
            os.chdir(src)
            for root, dirs, files in os.walk('.', followlinks=True):
                for _f in files:
                    try:
                        arc_fh.write(os.path.join(root, _f))
                        os.chmod(os.path.join(root, _f), 0o777)
                    except:
                        logger.info("Failed to read {}. Maybe file is a broken symlink?".format(_f))
                for _dir in dirs:
                    try:
                        os.chmod(os.path.join(root, _dir), 0o777)
                    except:
                        logger.info("Failed to add directory {}. Maybe directory is a broken symlink?".format(_dir))
            os.chdir(cwd)
        shutil.move(archive, os.path.join(archive_folder, archive))
        logger.info('Zipped {}'.format(src))
        logger.debug('Zipped archive folder {}, content: {}'.format(archive_folder, os.listdir(archive_folder)))

    @staticmethod
    def unpack(archive, folder):
        logger.info('Unzip {}'.format(archive))
        mode = 'r'
        try:
            with PatchedZipFile(archive, mode) as arc_fh:
                arc_fh.extractall(folder)
        except Exception as e:
            logger.exception(e)
        logger.info('Unzipped {} to {}'.format(archive, folder))
        logger.debug('Folder {} has unpacked content: {}'.format(folder, os.listdir(folder)))


ARCH_TYPES = {
    ".tar": ArchiveTar,
    ".tgz": ArchiveTar,
    ".zip": ArchiveZip,
}


class Archive(object):

    @staticmethod
    def arch_type(filename):
        try:
            file_ext = filename[len(filename)-4:]
            return ARCH_TYPES[file_ext]
        except KeyError as e:
            logger.exception("Unknown archive type for {}".format(filename))
            raise e

    @staticmethod
    def pack(archive, *args, **kwargs):
        Archive.arch_type(archive).pack(archive, *args, **kwargs)

    @staticmethod
    def unpack(archive, *args, **kwargs):
        Archive.arch_type(archive).unpack(archive, *args, **kwargs)
