import base64
import logging
import os
import random
import string
import tempfile

from sandbox.projects.common.decorators import retries

from sandbox.sandboxsdk.environments import SandboxEnvironment
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess

_logger = logging.getLogger('keychain')

CREATE_AND_ADD_KEYCHAIN = '''
#!/bin/bash -e

echo "New Keychain name: $keychain"

keychains=$(security list-keychains -d user)
keychainNames=();
for cur_keychain in $keychains
do
  basename=$(basename "$cur_keychain")
  keychainName=${basename::${#basename}-4}
  keychainNames+=("$keychainName")
done

echo "User keychains on this machine: ${keychainNames[@]}";
security create-keychain -p $pass $keychain
security list-keychains -s "${keychainNames[@]}" $keychain
security default-keychain -s $keychain
security unlock-keychain -p $pass $keychain
security set-keychain-settings -lut 7200 $keychain
'''

ADD_CERT = 'security import $cert_path -k $keychain -P $cert_pass -A'

SET_KEY_PARTITION_LIST = 'security set-key-partition-list -S apple-tool:,apple: -k $pass $keychain'


def random_string(string_length=8):
    letters = string.ascii_lowercase
    return ''.join(random.choice(letters) for i in range(string_length))


class MacOsKeychain(object):
    def __init__(self):
        self._env = {
            'keychain': random_string(),
            'pass': random_string()
        }

    def _write_to_tmp_file(self, data, mode, filename=None):
        temp = tempfile.NamedTemporaryFile().name
        if filename:
            temp = os.path.join(os.path.dirname(temp), filename)
        with open(temp, mode) as f:
            f.write(data)
        return temp

    @retries(5, exceptions=(subprocess.CalledProcessError,))
    def _execute(self, cmd):
        _logger.info('Start to execute {}.'.format(cmd))
        pc = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE, close_fds=True, env=self._env)
        out, err = pc.communicate()
        _logger.error('stdout: {} stderr: {}'.format(out, err))
        if pc.returncode != 0:
            raise Exception('Error while preparing keychain.')

    def create(self):
        create_file = self._write_to_tmp_file(CREATE_AND_ADD_KEYCHAIN, 'w')
        self._execute('source {}'.format(create_file))
        self._execute('security list-keychains')

    def add_cert(self, cert):
        cert_sec, cert_name = cert.split(':')
        cert_file = '{}.p12'.format(cert_name)
        cert_content_vault = '{}[{}]'.format(cert_sec, cert_file)
        cert_pass_vault = '{}[{}_pass]'.format(cert_sec, cert_name)
        cert_file = self._write_to_tmp_file(base64.b64decode(sdk2.Vault.data(cert_content_vault)), 'wb', cert_file)
        self._env['cert_path'] = cert_file
        self._env['cert_pass'] = sdk2.Vault.data(cert_pass_vault)
        self._execute(ADD_CERT)

    def set_key_partition_list(self):
        self._execute(SET_KEY_PARTITION_LIST)


class MacOsKeychainEnvironment(SandboxEnvironment):
    def __init__(self,
                 certs_pairs_vault):
        super(MacOsKeychainEnvironment, self).__init__()
        if not isinstance(certs_pairs_vault, list):
            raise Exception('certs_pairs_vault should be list, given type "{}"'.format(type(certs_pairs_vault)))
        self._certs_pairs_vault = certs_pairs_vault

    @classmethod
    def _keychains_dir(cls):
        return os.path.join(os.path.expanduser('~'), 'Library', 'Keychains')

    def prepare(self):
        if not self._certs_pairs_vault:
            _logger.info('Keychain preparing finished, since the passed certs are empty.')
        else:
            self._keychain = MacOsKeychain()
            self._keychain.create()
            for cert_vault_pair in self._certs_pairs_vault:
                self._keychain.add_cert(cert_vault_pair)
            self._keychain.set_key_partition_list()
