import os
import logging
import tempfile

from sandbox.common.errors import TaskFailure
from sandbox.sdk2.helpers import process

_logger = logging.getLogger(__name__)


def run(cmd, timeout):
    try:
        _logger.info('Start to execute {}.'.format(cmd))
        pc = process.subprocess.run(cmd, shell=True,
                                    check=True,
                                    stdout=process.subprocess.PIPE,
                                    stderr=process.subprocess.PIPE,
                                    timeout=timeout)
        _logger.info('{} executed.\n stdout: {}\n stderr: {}'.format(cmd, pc.stdout, pc.stderr))
    except (process.subprocess.CalledProcessError, process.subprocess.TimeoutExpired) as err:
        raise RuntimeError('{cmd} failed: {err}\n'.format(cmd=cmd, err=err))


def execute_on_platform(task, cmd, logger_name, platform, message=None, cwd=None):
    logger = logging.getLogger(logger_name)
    logger.info('execute {}.'.format(cmd))
    with process.ProcessLog(task, logger) as pl:
        with tempfile.NamedTemporaryFile(mode='w+x', prefix=logger_name) as cmd_tempfile:
            logger.debug("Execution platform: {}".format(platform))
            if platform == 'macos_arm':
                write_cmd_tempfile(cmd_tempfile, cmd)
                cmd = 'arch -arm64 sh -c {}'.format(cmd_tempfile.name)
                logger.info("on Arm execute via sh script with arch -arm64")
            _env = task._env if hasattr(task, '_env') else None
            rc = process.subprocess.Popen(
                args=cmd,
                shell=True,
                cwd=cwd,
                env=_env,
                stdout=pl.stdout,
                stderr=pl.stderr,
            ).wait()

            logger.info('executed {}'.format(cmd))
            if message and rc:
                raise TaskFailure(message)
            return rc


def execute(cmd, env=None, in_background=False, timeout=60):
    _logger.info('Start to execute {}.'.format(cmd))
    pc = process.subprocess.Popen(cmd, shell=True, stdout=process.subprocess.PIPE, stderr=process.subprocess.PIPE, close_fds=True, env=env)
    try:
        out, err = pc.communicate(timeout=timeout)
        _logger.info('stdout: {} stderr: {}'.format(out, err))
        if pc.returncode != 0:
            raise Exception('Error while executing {}. stdout: {}, stderr {}'.format(cmd, out, err))
    except process.subprocess.TimeoutExpired as e:
        if in_background:
            _logger.info('Continue command execution in background: {}.'.format(cmd))
        else:
            pc.kill()
            out, err = pc.communicate()
            _logger.error("Terminated after timeout {}, stdout: {} stderr: {}.".format(cmd, out, err))
            raise e


def detect_platform(rosetta=False):
    pc = process.subprocess.check_output(['uname', '-a']).split()
    arch = 'x86'
    os_family = 'UNKNOWN'
    for item in pc:
        if item == 'Linux':
            os_family = 'linux'
        if item == 'Darwin':
            os_family = 'macos'
        # uname -a with Rosetta returns "... RELEASE_ARM64_T8101 x86_64".
        if 'RELEASE_ARM64' in item:
            arch = 'arm'

    # Override detected platform
    if rosetta:
        _logger.debug('Override platform. Set forcefully x86')
        arch = 'x86'

    _logger.debug('detect_platform result: {}_{}. stdout: {}'.format(os_family, arch, pc))
    return '{}_{}'.format(os_family, arch)


def write_cmd_tempfile(cmd_tempfile, args):
    cmd_tempfile.write(args)
    cmd_tempfile.flush()
    os.chmod(cmd_tempfile.name, 0o744)
