# -*- coding: utf-8 -*-

import logging
import os
import shutil
import tempfile

from pathlib2 import Path

from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.projects.common import binary_task
from sandbox.projects.metrika.mobile.sdk.helpers.ShellExecutor import ShellExecutor
from sandbox.projects.mobile_apps.utils.resources import XcodeSimulatorArchive
from sandbox.projects.mobile_apps.utils.rvm_plus_ruby_env import RvmPlusRubyEnvironment
from sandbox.projects.mobile_apps.utils import shellexecuter as utils
from sandbox.sdk2.environments import Xcode


logger = logging.getLogger(__name__)

RUNTIME_DMG_DIRECTORY = os.path.expanduser("~/Library/Caches/XcodeInstall")


class XcodeSimulatorPreparerParameters(sdk2.Task.Parameters):
    ext_params = binary_task.binary_release_parameters(stable=True)

    with sdk2.parameters.Group("Xcode Simulator Preparer parameters") as params:

        xcode_version = sdk2.parameters.String(
            "Xcode version",
            required=True,
            default="13.2.1"
        )
        os_name = sdk2.parameters.String(
            "iOS/iPadOS name, i.e 'iOS 15.0'",
            required=True,
            default="iOS 15.0"
        )
        ruby_rvm_version = sdk2.parameters.String(
            "RVM_PLUS_RUBY resource version, i.e '1.29.12+3.0.2'",
            required=True,
            default="1.29.12+3.0.2"
        )
        disk_space = sdk2.parameters.Integer(
            'Required disk space in GB',
            default=50,
        )


class XcodeSimulatorPreparer(binary_task.LastBinaryTaskRelease, sdk2.Task):

    class Parameters(XcodeSimulatorPreparerParameters):
        pass

    class Requirements(sdk2.Requirements):
        cores = 2
        ram = 8192

    def on_save(self):
        self.Requirements.disk_space = self.Parameters.disk_space * 1024
        super(XcodeSimulatorPreparer, self).on_save()

    def on_prepare(self):
        logger.info("Started to prepare Xcode environment")
        xcode_version = self.Parameters.xcode_version
        self.shell = ShellExecutor()
        self.Context.platform = utils.detect_platform()

        with sdk2.helpers.ProgressMeter('Install Xcode {}'.format(xcode_version)):
            Xcode(xcode_version).prepare()
        logger.info("Prepared Xcode")

        logger.info("Started to prepare ruby+rvm environment")
        logger.debug("Detected platform: {}".format(self.Context.platform))
        rvm_ruby_environment = RvmPlusRubyEnvironment(version=self.Parameters.ruby_rvm_version,
                                                      platform=self.Context.platform)
        rvm_ruby_environment.prepare()
        logger.info("Prepared rvm+ruby")

        if os.path.exists(RUNTIME_DMG_DIRECTORY):
            if not os.listdir(RUNTIME_DMG_DIRECTORY):
                logger.info("Cleaning {}".format(RUNTIME_DMG_DIRECTORY))
                shutil.rmtree(RUNTIME_DMG_DIRECTORY)
                os.mkdir(RUNTIME_DMG_DIRECTORY)
                logger.info("Recreated {}".format(RUNTIME_DMG_DIRECTORY))

    def on_execute(self):
        logger.info("Task started.")
        logger.info("Parameters: {}".format(self.Parameters))
        os_name = self.Parameters.os_name

        with sdk2.helpers.ProgressMeter('Install xcode-install'):
            self.shell.execute_shell_and_check("gem install xcode-install")
        with sdk2.helpers.ProgressMeter('Install simulator {}'.format(os_name)):
            self.shell.execute_shell_and_check("xcversion simulators --no-install --install='{}'".format(os_name))

        with sdk2.helpers.ProgressMeter('Extract Runtime from dmg'):
            dmg_file = self._get_dmg_file()
            with Dmg(dmg_file) as dmg_dir:
                pkg_file = self._get_pkg_from_dmg(dmg_dir)
                content_path = self._extract_content_from_pkg(pkg_file)

            runtime_dir = "{}.simruntime".format(self.Parameters.os_name)
            runtime_full_dir = os.path.join(os.getcwd(), runtime_dir)
            os.mkdir(runtime_full_dir)
            shutil.move(content_path, runtime_full_dir)

        with sdk2.helpers.ProgressMeter('Create resource'):
            self._create_resource(runtime_full_dir)
        logger.info("Task finished.")

    def _create_resource(self, path):
        ttl = "inf" if (self.Parameters.ext_params.binary_executor_release_type == "stable") else "14"
        attrs = {
            "platform": self.Context.platform,
            "version": self.Parameters.os_name,
            "ttl": ttl
        }
        description = "OS name: {}, platform {}".format(self.Parameters.os_name,
                                                        self.Context.platform)
        sdk2.ResourceData(XcodeSimulatorArchive(self, description, path, **attrs)).ready()

    def _get_pkg_from_dmg(self, dmg_dir):
        return self._get_file_from_directory(dmg_dir, "*.pkg")

    def _get_dmg_file(self):
        return self._get_file_from_directory(RUNTIME_DMG_DIRECTORY, "*.dmg")

    def _get_file_from_directory(self, dir, pattern):
        """
            Function expects that directory has only one file with specified mask.
        """
        content = list(Path(dir).glob(pattern))
        if len(content) != 1:
            raise TaskError("{} should have one file with pattern {}, but found: {}".format(dir, pattern, content))
        return str(content.pop())

    def _extract_content_from_pkg(self, pkg_file):
        temp_extract_dir = tempfile.mkdtemp(prefix='pkg_content_', dir=os.getcwd())
        self.shell.execute_shell_and_check("xar -xf {}".format(pkg_file), cwd=temp_extract_dir)
        self.shell.execute_shell_and_check("cat ./Payload | gunzip -dc | cpio -i", cwd=temp_extract_dir)
        return os.path.join(temp_extract_dir, "Contents")


class Dmg:

    def __init__(self, dmg_file):
        self.dmg_file = dmg_file
        self.mount_point = 'mount_point_{}'.format(os.path.basename(dmg_file))
        self.shell = ShellExecutor()

    def __enter__(self):
        self.shell.execute_shell_and_check(
            ['hdiutil', 'attach', '-mountpoint', self.mount_point, self.dmg_file],
            logger_name='hdiutil_attach_{}'.format(os.path.basename(self.dmg_file))
        )
        return self.mount_point

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.shell.execute_shell_and_check(
            ['hdiutil', 'detach', '-force', self.mount_point],
            logger_name='hdiutil_detach_{}'.format(os.path.basename(self.dmg_file))
        )
