# coding: utf-8

import re
import os
import logging
import json
from sandbox import sdk2
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.sdk2.helpers.process import subprocess
from sandbox.common import errors as common_errors


class BadSecretsTemplates(Exception):

    def __init__(self, wrong_templates):
        self.wrong_templates = wrong_templates


class NoOccurrencesInLaunchScript(BadSecretsTemplates):
    pass


class WrongSecretMetadataFormat(BadSecretsTemplates):
    pass


class DuplicatedTemplates(BadSecretsTemplates):
    pass


class InaccessibleSecrets:
    not_found = list()
    not_delegated = list()


class InaccessibleSecretsError(Exception):

    def __init__(self, inaccessible_templates):
        self.inaccessible_templates = inaccessible_templates


def get_string_from_list(header, list_of_str):
    header = header + '\n'
    resulting_string = str()
    for template in list_of_str:
        resulting_string += template + '\n'
    return header + resulting_string


# initialized by dict of type [str: str]
#
# expects that value of each key can be evaluated as tuple containing 1 or 2 elements
# if tuple contains 2 elements the first one is recognized as a secret's key and the second as version of secret
# if tuple contains 1 element this elements is recognized as a secret's
#
# returns a dict of type [str: str] where key is template name and value is a secret value


class SecretManager:
    hash = '[0-9A-Za-z]'
    secret_key = '[0-9A-Za-z-_.]+'
    secret_id = 'sec-+' + hash + '+'
    version = 'ver-' + hash + '+'
    spaces = '\\s*'
    end = '$'
    or_re = '|'
    comma = ','
    id_and_ver = spaces + secret_id + comma + spaces + version + spaces + comma + spaces + secret_key + spaces + end
    id_only = spaces + secret_id + spaces + comma + spaces + secret_key + spaces + end
    full_re = re.compile(id_and_ver + or_re + id_only)

    @staticmethod
    def get_template_and_secret(serialized_info):
        secret_info = tuple(map(str, serialized_info.split(',')))
        vault_formatted_key = SecretManager.construct_vault_key(secret_info)
        return sdk2.Vault.data(vault_formatted_key)

    @staticmethod
    def construct_vault_key(secret_info):
        if len(secret_info) == 3:
            (secret_id, version, key) = secret_info
            return secret_id.strip() + '@' + version.strip() + '[' + key.strip() + ']'
        else:
            (secret_id, key) = secret_info
            return secret_id.strip() + '[' + key.strip() + ']'


class SecretsProcessor:
    def __init__(self, serialized_secrets_meta):
        self.serialized_secrets_meta = serialized_secrets_meta
        self.inaccessible_secrets = InaccessibleSecrets()

    def evaluate_secrets(self):
        wrong_templates = self.validate_meta()
        if len(wrong_templates) != 0:
            raise BadSecretsTemplates(wrong_templates=wrong_templates)

        secrets = dict()
        for template, meta in self.serialized_secrets_meta.items():
            try:
                secret = SecretManager.get_template_and_secret(meta)
                secrets[template] = secret
                logging.info('Secret for template: ' + template + ' has been received')
            except common_errors.VaultNotAllowed:
                self.inaccessible_secrets.not_delegated.append(template)
            except common_errors.VaultNotFound:
                self.inaccessible_secrets.not_found.append(template)

        if len(self.inaccessible_secrets.not_found) != 0 or len(self.inaccessible_secrets.not_delegated) != 0:
            raise InaccessibleSecretsError(inaccessible_templates=self.inaccessible_secrets)

        return secrets

    def validate_meta(self):
        wrong_templates = list()
        for template, meta in self.serialized_secrets_meta.items():
            if not re.match(SecretManager.full_re, meta):
                wrong_templates.append(template)
                logging.info("The secret meta for template: " + template + " is wrong")
        return wrong_templates

    # if secret meta is in wrong format SDK just fails execution using assert (as I understood from source code)


class WrongTemplates:
    duplicated = list()
    non_existing = list()


class TemplatesValidator:
    def __init__(self, templates, launch_script):
        self.templates = templates
        self.launch_script = launch_script

    def validate(self):
        wrong_templates = WrongTemplates()
        wrong_templates.duplicated = self.duplicated_templates()
        wrong_templates.non_existing = self.non_existing_templates_in_launch_script()
        return wrong_templates

    def duplicated_templates(self):
        non_unique_templates = set()
        template_set = set()
        for template in self.templates:
            prev_count = len(template_set)
            template_set.add(template)
            if prev_count == len(template_set):
                logging.info("Template:" + template + " isn't unique")
                non_unique_templates.add(template)
        if len(non_unique_templates) == 0:
            logging.info("All task's templates are unique")
        return non_unique_templates

    def non_existing_templates_in_launch_script(self):
        non_existing_templates = set()
        for template in self.templates:
            occurrences_count = self.launch_script.count(template)
            if occurrences_count == 0:
                logging.info("Template:" + template + " absent in launch script")
                non_existing_templates.add(template)
        return non_existing_templates


class SlackReporter:
    class SectionBlock:
        @staticmethod
        def get_markdown_text_block(text):
            block_dict = dict()
            block_dict["type"] = "section"
            text_dict = dict()
            text_dict["type"] = "mrkdwn"
            text_dict["text"] = text
            block_dict["text"] = text_dict
            return block_dict

        @staticmethod
        def get_message(blocks):
            message_dict = dict()
            message_dict["text"] = ""
            message_dict["blocks"] = blocks
            return message_dict

    slack_unprepared_script = "curl -X POST -H 'Content-type: application/json' --data '{0}' {1}"
    unprepared_url = "<https://sandbox.yandex-team.ru/task/{0}/view|View Sandbox Task>"

    def report(self, web_hook, text, task_id, tag):
        text_block = self.SectionBlock.get_markdown_text_block(text)
        tag_block = self.SectionBlock.get_markdown_text_block("Tag: " + tag)
        task_url_block = self.SectionBlock.get_markdown_text_block(self.unprepared_url.format(task_id))
        blocks = self.SectionBlock.get_message([text_block, tag_block, task_url_block])
        slack_prepared_script = self.slack_unprepared_script.format(
            json.dumps(blocks),
            web_hook
        )

        retry = 10
        while retry > 0:
            return_code = os.system(slack_prepared_script)
            if return_code == 0:
                break
            retry -= 1

        if retry == 0:
            logging.info("Can't report to Slack")


class SSHValidator:
    def __init__(self, ssh_meta):
        self.ssh_meta = ssh_meta

    def evaluate_ssh(self):
        if self.is_meta_valid():
            try:
                return SecretManager.get_template_and_secret(self.ssh_meta)
            except common_errors.VaultNotFound:
                return None
            except common_errors.VaultNotAllowed:
                return None
        else:
            return None

    def is_meta_valid(self):
        if not re.match(SecretManager.full_re, self.ssh_meta):
            logging.info("The SSH meta is wrong")
            return False
        return True


class MapsMobileTaskRunner(sdk2.Task):
    """ Task for running other tasks belong to Yandex.Maps, iOS team """
    secrets = dict()
    slack_reporter = SlackReporter()

    class Parameters(sdk2.Task.Parameters):
        # sets up built in parameters
        # launch script parameters
        secrets_meta = sdk2.parameters.Dict('Secrets [value format: id {sec-[0-9A-Za-z]+}, version {ver-[0-9A-Za-z]+}, key {[0-9A-Za-z-_.]+}]', required=False)
        launch_script = sdk2.parameters.String("Launch script", multiline=True, required=True)
        slack_service_web_hook = sdk2.parameters.String("Slack service webhook", required=True)
        working_directory = sdk2.parameters.String("Working directory (Arcadia)", required=False)
        ssh_key = sdk2.parameters.String('SSH key', required=False)
        tag = sdk2.parameters.String('Task tag', required=True)

    def on_timeout(self):
        message = 'Task was killed due to timeout'
        self.slack_reporter.report(
            web_hook=self.Parameters.slack_service_web_hook,
            text=message,
            task_id=self.id,
            tag=self.Parameters.tag
        )

    def on_failure(self):
        message = 'Task was finished with \"FAILURE\" status'
        self.slack_reporter.report(
            web_hook=self.Parameters.slack_service_web_hook,
            text=message,
            task_id=self.id,
            tag=self.Parameters.tag
        )

    def on_execute(self):
        # validate templates

        launch_script_validator = TemplatesValidator(
            templates=self.all_templates(),
            launch_script=self.Parameters.launch_script
        )
        wrong_templates = launch_script_validator.validate()
        if len(wrong_templates.duplicated) != 0:
            message = get_string_from_list(header='Duplicated templates', list_of_str=wrong_templates.duplicated)
            logging.info(message)
            self.slack_reporter.report(
                web_hook=self.Parameters.slack_service_web_hook,
                text=message,
                task_id=self.id,
                tag=self.Parameters.tag
            )
        if len(wrong_templates.non_existing) != 0:
            message = get_string_from_list(header='Non existing templates', list_of_str=wrong_templates.non_existing)
            logging.info(message)
            self.slack_reporter.report(
                web_hook=self.Parameters.slack_service_web_hook,
                text=message,
                task_id=self.id,
                tag=self.Parameters.tag
            )

        # validate secrets

        try:
            secrets_processor = SecretsProcessor(self.Parameters.secrets_meta)
            self.secrets = secrets_processor.evaluate_secrets()
        except BadSecretsTemplates as err:
            message = get_string_from_list(header='Wrong formatted secrets', list_of_str=err.wrong_templates)
            logging.info(message)
            self.slack_reporter.report(
                web_hook=self.Parameters.slack_service_web_hook,
                text=message,
                task_id=self.id,
                tag=self.Parameters.tag
            )
            return
        except InaccessibleSecretsError as err:
            message = str()
            if len(err.inaccessible_templates.not_delegated) != 0:
                message += get_string_from_list(header='Not delegated secrets', list_of_str=err.inaccessible_templates.not_delegated)
            elif len(err.inaccessible_templates.not_found) != 0:
                message += get_string_from_list(header='Not found secrets', list_of_str=err.inaccessible_templates.not_found)
            self.slack_reporter.report(
                web_hook=self.Parameters.slack_service_web_hook,
                text=message,
                task_id=self.id,
                tag=self.Parameters.tag
            )
            return

        if len(wrong_templates.duplicated) != 0 or len(wrong_templates.non_existing) != 0:
            return

        # groups all templates with value
        prepared_launch_script = self.get_prepared_launch_script()
        with arcadia_sdk.mount_arc_path('arcadia-arc:/#trunk') as arcadia_path:
            if len(self.Parameters.ssh_key.strip()) == 0:
                self.execute_command(arcadia_path, prepared_launch_script)
                return
            ssh_private_part = SSHValidator(self.Parameters.ssh_key).evaluate_ssh()
            if ssh_private_part is None:
                message = 'Wrong formatted ssh'
                logging.info(message)
                self.slack_reporter.report(
                    web_hook=self.Parameters.slack_service_web_hook,
                    text=message,
                    task_id=self.id,
                    tag=self.Parameters.tag
                )
                return
            else:
                with sdk2.ssh.Key(private_part=ssh_private_part):
                    self.execute_command(arcadia_path, prepared_launch_script)

    def execute_command(self, arcadia_path, prepared_launch_script):
        logging.info('Execute command')
        cmd_logger = logging.getLogger('cmd')

        with sdk2.helpers.ProcessLog(self, logger=cmd_logger) as process_log:
            return_code = subprocess.Popen(
                prepared_launch_script,
                shell=True,
                stdout=process_log.stdout,
                stderr=process_log.stdout,
                cwd=arcadia_path + "/" + self.Parameters.working_directory.strip()
            ).wait()

            if return_code != 0:
                error_message = 'Command died with exit code ' + str(return_code)
                logging.info(error_message)
                self.slack_reporter.report(
                    web_hook=self.Parameters.slack_service_web_hook,
                    text=error_message,
                    task_id=self.id,
                    tag=self.Parameters.tag
                )
                return
        logging.info('Command executed.')

    def get_prepared_launch_script(self):
        prev_len = len(self.Parameters.launch_script)
        prepared_launch_script = self.Parameters.launch_script
        # вставляем id сэндбоксовой таски если нужно
        prepared_launch_script = prepared_launch_script.replace('SANDBOX_TASK_ID', str(self.id))
        # вставляем секреты
        for template, value in self.secrets.items():
            prepared_launch_script = prepared_launch_script.replace(template, value)
        new_len = len(self.Parameters.launch_script)
        logging.info("length of launch script has been changed from " + str(prev_len) + " to " + str(new_len))
        return prepared_launch_script

    def all_templates(self):
        return list(self.Parameters.secrets_meta.keys())
