# -*- coding: utf-8 -*-
import os

import subprocess
import sandbox.common.types.misc as ctm
import logging
from sandbox import sdk2
from sandbox.projects.metrika.utils.mixins.console import BaseConsoleMixin
from sandbox.projects.metrika.utils.vcs import checkout
from sandbox.sandboxsdk import environments
from sandbox.projects.metrika.utils.parameters import VcsParameters, LastPeasantContainerResource
from sandbox.projects.mobile_apps.utils.rvm_plus_ruby_env import RvmPlusRubyEnvironment

_logger = logging.getLogger('MobileadsReleaseScriptJob')


class MobileadsReleaseScriptJob(sdk2.Task, BaseConsoleMixin):
    WORKING_DIR = 'wd'

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        container = LastPeasantContainerResource('Environment container resource', required=True)
        kill_timeout = 36000
        vcs_block = VcsParameters()

        vault_owner = sdk2.parameters.String('Private key vault onwer',
                                             description='Имя группы/пользователя, владельца секрета, '
                                                         'в котором хранится приватный ключ',
                                             required=True,
                                             default='YANDEX_MOBILEADS_SDK')

        private_key_vault_name = sdk2.parameters.String('Private key vault name',
                                                        description='Имя секрета, '
                                                                    'в котором хранится приватный ssh ключ',
                                                        required=True)

        script = sdk2.parameters.String('Script to run', required=True)

        with sdk2.parameters.Group('Ruby (only if ios)') as ruby:
            ruby_env_needed = sdk2.parameters.Bool('Ruby env needed', default=False)
            ruby_env_linux = sdk2.parameters.Bool('Ruby env linux', default=True)
            ruby_env_version = sdk2.parameters.String('Rvm and ruby version', default='1.29.10+2.7.0')

        with sdk2.parameters.Group('Tokens') as reporting:
            startrek_token_vault_name = sdk2.parameters.String('Startrek token vault name',
                                                               description='Имя секрета, '
                                                                           'в котором хранится startrek-token',
                                                               required=False)

            bitbucket_token_vault_name = sdk2.parameters.String('Bitbucket token vault name',
                                                                description='Имя секрета, '
                                                                            'в котором хранится bitbucket-token',
                                                                required=False)

            testpalm_token_vault_name = sdk2.parameters.String('Testpalm token vault name',
                                                               description='Имя секрета, '
                                                                           'в котором хранится testpalm-token',
                                                               required=False)

            teamcity_token_vault_name = sdk2.parameters.String('Teamcity token vault name',
                                                               description='Имя секрета, '
                                                                           'в котором хранится teamcity-token',
                                                               required=False)

            hitman_token_vault_name = sdk2.parameters.String('Hitman token vault name',
                                                             description='Имя секрета, '
                                                                         'в котором хранится hitman-token',
                                                             required=False)
        with sdk2.parameters.Group('Update sdk versions') as versions:
            sdk_name = sdk2.parameters.String('Sdk name',
                                              description='admob|applovin|ironsource|facebook|mopub|mytarget|unityads')
            branch_to = sdk2.parameters.String('Branch for release pr',
                                               description='Ветка в которую создавать ПР с изменениями',
                                               default='rc-adapters')

        with sdk2.parameters.Group('Pull requests') as prs:
            repo = sdk2.parameters.String('Repo name',
                                          description='Название репозитория',
                                          default='mobile-ads-library-android',
                                          required=False)
            pr_from = sdk2.parameters.String('PR from',
                                             description='Ветка, из которой создавать ПР')
            pr_to = sdk2.parameters.String('PR to',
                                           description='Маски веток через запятую, в которые создавать ПР',
                                           default='dev.*,rc.*,feature.*')

        with sdk2.parameters.Group('Autotests versions') as autotests:
            autotests_pr_to = sdk2.parameters.String('Pr branch destination',
                                                     default='dev')
            update_pods = sdk2.parameters.Bool('Update pods',
                                               default=True,
                                               description='Обновлять поды ios-агента')
            autotests_issue = sdk2.parameters.String('Autotests ST issue',
                                                     description='ID задачи')

        with sdk2.parameters.Group('Teamcity config') as teamcity:
            build = sdk2.parameters.String('Build name',
                                           description='Имя билда')
            build_branch = sdk2.parameters.String('Branch',
                                                  description='Ветка, из которой билдить')
            upload_branch = sdk2.parameters.String('Upload branch',
                                                   description='Ветка, куда заливать sample')

        with sdk2.parameters.Group('Assessors testing') as tesing:
            platform = sdk2.parameters.String('Platform',
                                              description='iOS/Android')
            app_name = sdk2.parameters.String('Beta app name',
                                              default='mobileads')
            testplans = sdk2.parameters.String('Testplan ids',
                                               description='id тестпланов через запятую')
            login = sdk2.parameters.String('Requester',
                                           description='Ваш логин')
            testpalm_version_name = sdk2.parameters.String('Version Name',
                                                           description='Имя версии (если не указано, будет стандартное)')
            speed = sdk2.parameters.String('Testing speed',
                                           description='Скорость тестирования (можно поставить URGENT)',
                                           default='NORMAL')
            overlap = sdk2.parameters.String("Number of overlaps",
                                             default="3",
                                             description="Indicates how much runs for each test suite will be created",
                                             required=True)
            booking_id = sdk2.parameters.String('Booking id',
                                                description='Указать, если уже есть')
            run_teamcity_build = sdk2.parameters.Bool('Run teamcity build',
                                                      default=True,
                                                      description='Запускать билд на teamcity')

    def on_execute(self):
        with sdk2.ssh.Key(self, self.Parameters.vault_owner, self.Parameters.private_key_vault_name):
            with sdk2.helpers.ProgressMeter('Checkout'):
                self._checkout_branch()
            with sdk2.helpers.ProgressMeter('Prepare ruby'):
                self.__prepare_ruby_env()
            with sdk2.helpers.ProgressMeter('Prepare environment'):
                self.__configure_environment()
            with sdk2.helpers.ProgressMeter('Run script'):
                self.__execute_script()

    def __prepare_ruby_env(self):
        if self.Parameters.ruby_env_needed:
            platform = 'linux' if self.Parameters.ruby_env_linux else 'darwin'
            self.__rvm_ruby_environment = RvmPlusRubyEnvironment(self.Parameters.ruby_env_version, platform=platform)
            self.__rvm_ruby_environment.prepare()

    def __execute_script(self):
        with environments.VirtualEnvironment() as venv:
            self.__install_requirements(venv)
            arguments = self._get_arguments()

            cmd = [venv.executable]

            if self.Parameters.script.endswith('.py'):
                cmd.append(os.path.join(self._work_dir(), self.Parameters.script))
            else:
                cmd.extend(['-m', self.Parameters.script])

            self.__run_command(cmd + arguments, 'script-log')

    def __install_requirements(self, venv):
        venv.pip('pip==19.1 setuptools==41.0.1')
        venv.pip('requests==2.23.0')
        venv.pip('gitpython==2.1.15')

    def __run_command(self, cmd, log_name):
        with sdk2.helpers.ProcessLog(self, logger=log_name) as pl:
            subprocess.check_call(
                cmd,
                stdout=pl.stdout,
                stderr=pl.stdout,
                cwd=self._work_dir(),
                env=self.__env)

    def _get_arguments(self):
        arguments = []
        if self.Parameters.sdk_name:
            arguments.extend(["--sdk_name", self.Parameters.sdk_name])
        if self.Parameters.branch_to:
            arguments.extend(["--branch_to", self.Parameters.branch_to])
        if self.Parameters.repo:
            arguments.extend(["--repo", self.Parameters.repo])
        if self.Parameters.pr_from:
            arguments.extend(["--pr_from", self.Parameters.pr_from])
        if self.Parameters.pr_to:
            arguments.extend(["--pr_to", self.Parameters.pr_to])
        if self.Parameters.platform:
            arguments.extend(["--platform", self.Parameters.platform])
        if self.Parameters.app_name:
            arguments.extend(["--app_name", self.Parameters.app_name])
        if self.Parameters.testplans:
            arguments.extend(["--testplans", self.Parameters.testplans])
        if self.Parameters.testpalm_version_name:
            arguments.extend(["--testpalm_version_name", self.Parameters.testpalm_version_name])
        if self.Parameters.login:
            arguments.extend(["--login", self.Parameters.login])
        if self.Parameters.speed:
            arguments.extend(["--speed", self.Parameters.speed])
        if self.Parameters.overlap:
            arguments.extend(["--overlap", self.Parameters.overlap])
        if self.Parameters.booking_id:
            arguments.extend(["--booking_id", self.Parameters.booking_id])
        if self.Parameters.build:
            arguments.extend(["--build", self.Parameters.build])
        if self.Parameters.run_teamcity_build is not None:
            arguments.extend(["--run_teamcity_build", str(self.Parameters.run_teamcity_build)])
        if self.Parameters.build_branch:
            arguments.extend(["--build_branch", self.Parameters.build_branch])
        if self.Parameters.upload_branch:
            arguments.extend(["--upload_branch", self.Parameters.upload_branch])
        if self.Parameters.autotests_pr_to:
            arguments.extend(["--autotests_pr_to", self.Parameters.autotests_pr_to])
        if self.Parameters.autotests_issue:
            arguments.extend(["--autotests_issue", self.Parameters.autotests_issue])
        if self.Parameters.update_pods is not None:
            arguments.extend(["--update_pods", str(self.Parameters.update_pods)])
        return arguments

    def __configure_environment(self):
        self.__env = os.environ.copy()
        if self.Parameters.startrek_token_vault_name:
            self.__env['STARTREK_TOKEN'] = sdk2.Vault.data(self.Parameters.vault_owner,
                                                           self.Parameters.startrek_token_vault_name)

        if self.Parameters.bitbucket_token_vault_name:
            self.__env['BITBUCKET_TOKEN'] = sdk2.Vault.data(self.Parameters.vault_owner,
                                                            self.Parameters.bitbucket_token_vault_name)

        if self.Parameters.testpalm_token_vault_name:
            self.__env['TESTPALM_TOKEN'] = sdk2.Vault.data(self.Parameters.vault_owner,
                                                           self.Parameters.testpalm_token_vault_name)

        if self.Parameters.teamcity_token_vault_name:
            self.__env['TEAMCITY_TOKEN'] = sdk2.Vault.data(self.Parameters.vault_owner,
                                                           self.Parameters.teamcity_token_vault_name)

        if self.Parameters.hitman_token_vault_name:
            self.__env['HITMAN_TOKEN'] = sdk2.Vault.data(self.Parameters.vault_owner,
                                                         self.Parameters.hitman_token_vault_name)
        _logger.info('Prepared environment: {}'.format(self.__env))

    def on_success(self, prev_status):
        pass

    def on_failure(self, prev_status):
        pass

    def _checkout_branch(self):
        checkout(self.Parameters.vcs_block, self._work_dir())

    def _work_dir(self, *path):
        return str(self.path(self.WORKING_DIR, *path))
