# -*- coding: utf-8 -*-
import sandbox.sdk2 as sdk2
import sandbox.projects.common.build.parameters
from sandbox.projects.common.build import YaPackage
from sandbox.projects.modadvert.BuildTask import ModAdvertBuildTask
from sandbox.projects.modadvert.common import modadvert
from sandbox.projects.modadvert.resource_types import (
    MODADVERT_LAAS_HTTP_RUNNER_BINARY, MODADVERT_HTTP_ENVIRONMENT_IMAGE,
    MODADVERT_LAAS_IN_MEMORY_RUNNER_BINARY, MODADVERT_LAAS_MR_RUNNER_BINARY
)


class BuildArtifact:
    def __init__(self, package_path, resource_class, build):
        self.package_path = package_path
        self.resource_class = resource_class
        self.build = build


class ModadvertBuildLaaS(modadvert.ModadvertBaseBinaryTask):
    """
    Builds specified LaaS runners (in-memory, HTTP, MR) and docker image with ready-to-use environment for HTTP runner.
    You need to specify arcadia path to YAML config with workers and path to JSON with list of required peerdirs.
    Note that you may not specify path to peerdirs if you don't use any non-standard libraries.
    """

    name = 'MODADVERT_BUILD_LAAS'

    class Parameters(modadvert.ModadvertBaseBinaryTask.Parameters):
        """
        Use options `build_in_memory_runner`, `build_http_runner`, `build_mr_runner` to specify which kind of runners you want to build.
        Use option `build_docker_image` to specify whether to build docker image or not.
        """

        kill_timeout = 12 * 60 * 60  # 12h

        with sdk2.parameters.Group('LaaS configuration'):
            workers_path = sdk2.parameters.String('Arcadia path to YAML config with workers config', required=True)
            peerdirs_path = sdk2.parameters.String('Arcadia path to JSON config with peerdirs', required=False)

        with sdk2.parameters.Group('Runners'):
            build_in_memory_runner = sdk2.parameters.Bool('Build in-memory runner', default=True)
            build_http_runner = sdk2.parameters.Bool('Build HTTP runner', default=True)
            build_mr_runner = sdk2.parameters.Bool('Build MR runner', default=True)
            build_docker_image = sdk2.parameters.Bool('Build Docker image for HTTP runner', default=True)

        with sdk2.parameters.Group('Build configuration'):
            checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl()
            use_aapi_fuse = sdk2.parameters.Bool('Use arcadia-api fuse', default=True)

        with sdk2.parameters.Group('Release configuration'):
            component_name = sdk2.parameters.String(ModAdvertBuildTask.ComponentName.description, required=False)
            release_number = sdk2.parameters.Integer(ModAdvertBuildTask.ReleaseNumber.description, required=False)
            resource_attributes = sdk2.parameters.Dict('Package resource attributes', required=False)

    def _create_build_parameters(self):
        from sandbox.projects.modadvert.BuildLaaS.patch import create_patch

        patch = create_patch(
            arcadia_url=self.Parameters.checkout_arcadia_from_url,
            build_paths=['modadvert/nirvana/laas/runner', 'modadvert/programs/laas/http_runner', 'modadvert/programs/laas/mr_runner'],
            workers_path=self.Parameters.workers_path,
            peerdirs_path=self.Parameters.peerdirs_path
        )
        build_artifacts = [
            BuildArtifact('modadvert/nirvana/laas/runner/package.json', MODADVERT_LAAS_IN_MEMORY_RUNNER_BINARY, self.Parameters.build_in_memory_runner),
            BuildArtifact('modadvert/programs/laas/http_runner/package.json', MODADVERT_LAAS_HTTP_RUNNER_BINARY, self.Parameters.build_http_runner),
            BuildArtifact('modadvert/programs/laas/mr_runner/package.json', MODADVERT_LAAS_MR_RUNNER_BINARY, self.Parameters.build_mr_runner),
            BuildArtifact('modadvert/docker/http_environment/package.json', MODADVERT_HTTP_ENVIRONMENT_IMAGE, self.Parameters.build_docker_image),
        ]
        packages = []
        resource_types = []
        resource_ids = []
        for artifact in build_artifacts:
            if not artifact.build:
                continue
            packages.append(artifact.package_path)
            resource = artifact.resource_class(
                task=self,
                description=artifact.package_path,
                path=artifact.resource_class.__name__.lower(),
                **self.Parameters.resource_attributes
            )
            resource_types.append(str(resource.type))
            resource_ids.append(str(resource.id))
        return {
            sandbox.projects.common.build.parameters.ArcadiaUrl.name: self.Parameters.checkout_arcadia_from_url,
            YaPackage.PackagesParameter.name: ';'.join(packages),
            YaPackage.ResourceTypeParameter.name: ';'.join(resource_types),
            sandbox.projects.common.build.parameters.ArcadiaPatch.name: patch,
            YaPackage.PackageTypeParameter.name: YaPackage.TARBALL,
            YaPackage.ResourceIdParameter.name: ';'.join(resource_ids),
            ModAdvertBuildTask.ComponentName.name: self.Parameters.component_name,
            ModAdvertBuildTask.ReleaseNumber.name: self.Parameters.release_number,
            YaPackage.PackageResourceAttrsParameter.name: self.Parameters.resource_attributes,
            sandbox.projects.common.build.parameters.UseArcadiaApiFuse.name: self.Parameters.use_aapi_fuse,
        }

    def on_execute_inner(self):
        if not self.Context.build_subtask:
            subtask_parameters = self._create_build_parameters()
            self.Context.build_subtask = self.create_subtask(ModAdvertBuildTask.type, subtask_parameters)
        self.wait_all_subtasks()
