# -*- coding: utf-8 -*-

import datetime
import logging

import sandbox.common.types.client as ctc
import sandbox.sandboxsdk.task as sdk_task
import sandbox.sandboxsdk.parameters as sdk_parameters
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.channel import channel
from sandbox.common import errors


class ModAdvertCheckNewCircuit(sdk_task.SandboxTask):
    type = "MODADVERT_CHECK_NEW_CIRCUIT"
    client_tags = ctc.Tag.LINUX_PRECISE

    class YtProxyUrl(sdk_parameters.SandboxStringParameter):
        name = "yt_proxy_url"
        description = name
        default_value = "hahn.yt.yandex.net"

    class VaultUser(sdk_parameters.SandboxStringParameter):
        name = "vault_user"
        description = name
        default_value = "MODADVERT"

    class YtTokenVaultName(sdk_parameters.SandboxStringParameter):
        name = "yt_token_vault_name"
        description = name
        default_value = "yt-token"

    class ResultTable(sdk_parameters.SandboxStringParameter):
        name = "result_table"
        description = 'Table with lyncher results'
        default_value = "//home/modadvert/lyncher/Results"

    class SampleSize(sdk_parameters.SandboxIntegerParameter):
        name = 'sample_size'
        description = 'Count of random banners to be considered'
        default_value = 50

    class Recipients(sdk_parameters.SandboxStringParameter):
        name = 'recps'
        description = 'Comma-separated list of recipients (without @yandex-team.ru)'
        default_value = 'iakolzin'

    # TODO: Use it or lose it
    '''
    class StatePath(sdk_parameters.SandboxStringParameter):
        name = "state_path"
        description = 'Cypress path with last successful launch'
        default_value = 10
    '''

    input_parameters = [
        YtProxyUrl,
        VaultUser,
        YtTokenVaultName,

        ResultTable,
        SampleSize,
        Recipients,
    ]

    environment = (
        environments.PipEnvironment('yandex-yt', '0.8.38a1', use_wheel=True),
        environments.PipEnvironment('yandex-yt-yson-bindings-skynet', use_wheel=True),
    )

    def on_execute(self):
        from . import yt_runner

        logging.info('starting ModAdvertCheckNewCircuit on_execute()...')

        if int(datetime.datetime.now().hour) <= 2:
            raise Exception('Too early to process')

        yt_token = self.get_vault_data(self.ctx.get(self.VaultUser.name), self.ctx.get(self.YtTokenVaultName.name))

        recps = self.ctx.get(self.Recipients.name).split(',')
        mail_list = [recp.replace(' ', '') + '@yandex-team.ru' for recp in recps]
        if not mail_list:
            logging.error('Can\'t determine mail list! Probably, it is empty, anyway, check it out: \'{}\''.format(self.ctx.get(self.Recipients.name)))
            return
        logging.info('Found mail-list: ' + str(mail_list))

        analyze_date = str(datetime.datetime.now() - datetime.timedelta(days=1))[:10]

        try:
            message = yt_runner.run(
                yt_proxy_url=self.ctx.get(self.YtProxyUrl.name),
                yt_token=yt_token,
                result_table=self.ctx.get(self.ResultTable.name),
                sample_size=str(self.ctx.get(self.SampleSize.name)),
                analyze_date=analyze_date,
            )
        except Exception as e:
            raise errors.TemporaryError("yt_runner failed with error: {}".format(e))

        logging.info('Got message ' + message)

        channel.sandbox.send_email(
            mail_to=mail_list,
            mail_cc=[],
            mail_subject='New circuit filters daily report for {}'.format(analyze_date),
            mail_body=message,
            content_type='text/html',
        )

        logging.info('finished ModAdvertCheckNewCircuit on_execute()')


__Task__ = ModAdvertCheckNewCircuit
