# -*- coding: utf-8 -*-
import logging
import os
from sandbox import sdk2
from sandbox.common.types import resource as ctr
from sandbox.projects.modadvert.ModadvertDeployTask import ModadvertDeployTask
from sandbox.projects.modadvert.resource_types import MODADVERT_LAAS_HTTP_RUNNER_BINARY
from sandbox.projects.modadvert.rm.constants import CUPID_OBJECT_TYPES
from sandbox.projects.release_machine.components.configs.cupid import CupidCfg


class ModadvertDeployCupid(ModadvertDeployTask):

    ADDITIONAL_PARAMETERS = ['http_runners_binaries']
    COMPONENT = CupidCfg.name
    ENV_NAMES = {
        'TESTING': 'testing',
        'PRODUCTION': 'production',
    }
    SCHEDULERS = {}

    @property
    def RELEASE_COMPONENTS(self):
        # Qloud doesn't accept component' names with underscores. So we have "display-href" component instead of "display_href"
        return [
            object_type.replace('_', '-')
            for object_type in self.Parameters.http_runners_binaries.keys()
        ]

    class Parameters(ModadvertDeployTask.Parameters):
        with sdk2.parameters.String('Release environment', multiline=True) as release_environment:
            release_environment.values.TESTING = release_environment.Value('TESTING', default=True)
            release_environment.values.PRODUCTION = release_environment.Value('PRODUCTION')

        with sdk2.parameters.Group('Resources') as binary_group:
            http_runners_binaries = sdk2.parameters.Dict(
                'Named list with resources of type MODADVERT_LAAS_HTTP_RUNNER_BINARY (key = object_type, value = resource_id)',
                required=True
            )

    def _untar_http_runner(self, object_type, resource):
        self.untar_resource(resource)
        resource_path = '{}_laas_http_runner'.format(object_type)
        self.run_command(['mv', 'laas_http_runner', resource_path])
        executable_resource = MODADVERT_LAAS_HTTP_RUNNER_BINARY(
            task=self,
            description='{} executable'.format(object_type),
            path=resource_path,
            object_type=object_type
        )
        sdk2.ResourceData(executable_resource).ready()
        return executable_resource

    def validate(self):
        super(ModadvertDeployCupid, self).validate()
        for object_type, resource_id in self.Parameters.http_runners_binaries.items():
            if object_type not in CUPID_OBJECT_TYPES:
                raise ValueError('Unknown object type {}. It should be one of {}'.format(object_type, CUPID_OBJECT_TYPES))
            if sdk2.Resource.find(id=resource_id, type=MODADVERT_LAAS_HTTP_RUNNER_BINARY, state=ctr.State.READY).first() is None:
                raise ValueError('Resource with id {} of type {} in state {} is not found'.format(resource_id, MODADVERT_LAAS_HTTP_RUNNER_BINARY, ctr.State.READY))

    def on_before_execute(self):
        super(ModadvertDeployCupid, self).on_before_execute()
        if not self.Context.executable_http_runners:
            self.Context.executable_http_runners = {}
            for object_type, resource_id in self.Parameters.http_runners_binaries.items():
                resource = sdk2.Resource.find(id=resource_id, type=MODADVERT_LAAS_HTTP_RUNNER_BINARY, state=ctr.State.READY).first()
                resource_data = sdk2.ResourceData(resource)
                if os.access(resource_data.path.as_posix(), os.X_OK):
                    logging.info('Executable HTTP runner for {} is found'.format(object_type))
                    self.Context.executable_http_runners[object_type] = resource.id
                else:
                    logging.info('Non-executable HTTP runner for {} is found. Unpacking...'.format(object_type))
                    self.Context.executable_http_runners[object_type] = self._untar_http_runner(object_type, resource).id

    def get_deploying_resources(self):
        return [
            '{}:/sandbox_resources/http_server:{}'.format(release_component, self.Context.executable_http_runners[release_component.replace('-', '_')])
            for release_component in self.RELEASE_COMPONENTS
        ]
