import logging
from collections import defaultdict

from sandbox import sdk2
import sandbox.common.types.resource as ctr
from sandbox.projects.modadvert.common import modadvert
from sandbox.sdk2.vcs.svn import Arcadia

from .utils import update_ydeploy_stage


class ModadvertDeployYDeployEnvironment(modadvert.ModadvertBaseTask):
    """
    MODDEV-2517: Deploy to YDeploy.
    """

    class Parameters(modadvert.ModadvertBaseTask.Parameters):
        with sdk2.parameters.Group('Credentials') as credentials_group:
            vault_user = sdk2.parameters.String('Vault user', default='MODADVERT')
            yp_token_vault_name = sdk2.parameters.String('YP token vault name', default='yp-token')
            sandbox_token_vault_name = sdk2.parameters.String('Sandbox token vault name', default='sandbox-token')
            commit_message = sdk2.parameters.String('Commit message', default='')

        with sdk2.parameters.Group('Resources') as resources_group:
            resources = sdk2.parameters.List(
                'List of deploying resources in format "component_name:resource_link:resource_id"',
                default=[]
            )
            resources_versions = sdk2.parameters.Dict(
                'Allow to update resources by its types regardless of the component. Format: "RESOURCE_TYPE:version"',
                default={}
            )

        with sdk2.parameters.Group('Deployment parameters') as deployment_group:
            environment_id = sdk2.parameters.String('Environment full ID', default='modadvert-supermoderation-dev')
            components = sdk2.parameters.Dict('Components (key = component name, value = docker image version)', default={})

    def create_env(self):
        return {
            'DCTL_YP_TOKEN': self.get_author_token(self.Parameters.yp_token_vault_name, vault_user=self.Parameters.vault_user),
            'SANDBOX_TOKEN': self.get_author_token(self.Parameters.sandbox_token_vault_name, vault_user=self.Parameters.vault_user),
        }

    def _get_version_by_resource_by_component(self):
        version_by_resource_by_component = defaultdict(dict)
        for resource in self.Parameters.resources:
            try:
                component, resource_link, resource_id = resource.split(':')
            except ValueError:
                raise ValueError('Invalid resource format {}. Expected "component_name:resource_link:resource_id"'.format(resource))
            if sdk2.Resource.find(id=resource_id, state=ctr.State.READY).first() is None:
                raise ValueError('Resource with id {} in {} state is not found'.format(resource_id, ctr.State.READY))
            resource_link = resource_link.strip('/').replace('/', '.')
            version_by_resource_by_component[component][resource_link] = resource_id
        return version_by_resource_by_component

    def on_execute_inner(self):

        def _popen_caller(command, env, stdout=None):
            logging.info('Running command: %s', command)
            with sdk2.helpers.ProcessRegistry, sdk2.helpers.ProcessLog(self, logger='main') as pl:
                sdk2.helpers.subprocess.check_call(command, stdout=stdout or pl.stdout, stderr=pl.stderr, env=env)
                pl.logger.info('Subprocess has finished successfully')

        Arcadia.export(url='arcadia:/arc/trunk/arcadia/ya', path='ya')

        update_ydeploy_stage(
            stage_name=self.Parameters.environment_id,
            components_with_versions=self.Parameters.components,
            version_by_resource_by_component=self._get_version_by_resource_by_component(),
            version_by_resource_type=self.Parameters.resources_versions,
            description=self.Parameters.commit_message,
            popen_caller=_popen_caller,
            env=self.create_env(),
            ya_path='./ya',
        )
