import codecs
import copy
import functools
import logging
import operator
import six
import subprocess
import yaml


def compose(funcs):
    return functools.partial(six.moves.reduce, lambda arg, func: func(arg), funcs)


def itemgetter_chain(*path):
    return compose(tuple(operator.itemgetter(key) for key in path))


class CompositeError(Exception):
    pass


class UpdateSpec(object):

    # We will use those paths to find and update resources.
    STATIC_RESOURCES_CONF_POSSIBLE_PATHS = (
        itemgetter_chain('multi_cluster_replica_set', 'replica_set',
                         'pod_template_spec', 'spec', 'pod_agent_payload', 'spec', 'resources', 'static_resources'),
        itemgetter_chain('replica_set', 'replica_set_template',
                         'pod_template_spec', 'spec', 'pod_agent_payload', 'spec', 'resources', 'static_resources'),
    )

    @classmethod
    def update_images(cls, spec, components_with_versions):
        """
        Update docker images versions for given components.
        :param spec: YDeploy config to update.
        :type spec: Dict[str, Any]
        :param components_with_versions: Deploy units names with images versions.
        :type components_with_versions: Dict[str, str]
        :return: None

        Example:
        >>> UpdateSpec.update_images(spec, {'api': '0.205.3', 'admin': '0.205.3', 'direct-proxy': '0.205.3'})
        """
        logging.info(u'%s.update_images(spec, `%s`)', cls.__name__, components_with_versions)
        errors = []
        deploy_units = spec['spec']['deploy_units']
        for component_name, version in components_with_versions.items():
            if component_name not in deploy_units:
                logging.warn('Not found component "%s" among deploy units. Skipping...', component_name)
                continue
            try:
                image_conf = deploy_units[component_name]['images_for_boxes'][component_name]
                initial = dict(image_conf)
                image_conf['tag'] = str(version)
                image_conf.pop('digest', None)
                logging.info('Image updated for %s. Before: `%s`. After: `%s`', component_name, initial, image_conf)
            except Exception as exc:
                logging.exception('Got exception while updating component %s', component_name)
                errors.append(exc)
        if errors:
            raise CompositeError(errors)

    @classmethod
    def _get_resources_items(cls, deploy_unit_conf):
        """
        :param deploy_unit_conf: YDeploy deploy unit conf.
        :type deploy_unit_conf: Dict[str, Any]
        :return: Dict[ResourceIdStr, Any]
        """
        res = {}
        for static_resources_getter in cls.STATIC_RESOURCES_CONF_POSSIBLE_PATHS:
            try:
                static_resources_conf = static_resources_getter(deploy_unit_conf)
            except KeyError:
                static_resources_conf = ()
            for resource_item in static_resources_conf:
                res[resource_item['id']] = resource_item
        return res

    @classmethod
    def _update_resource_item(cls, resource_item, new_version):
        """
        :param resource_item: Resource info item.
        :type resource_item: Dict[str, Any]
        :param new_version: Sandbox resource version.
        :type new_version: str
        :return: None
        """
        initial = copy.deepcopy(resource_item)
        if resource_item['url'].startswith('sbr:'):
            resource_item['url'] = 'sbr:{}'.format(new_version)
        else:
            raise ValueError(u'Cant update resource without sandbox version: {}'.format(resource_item))
        if (resource_item.get('verification') or {}).get('checksum'):
            resource_item['verification']['checksum'] = 'EMPTY:'
        logging.info(u'Resource updated. Before: `%s`. After: `%s`', initial, resource_item)

    @classmethod
    def update_resources(cls, spec, version_by_resource_by_component):
        """
        Update resources for given components.
        :param spec: YDeploy config to update.
        :type spec: Dict[str, Any]
        :param version_by_resource_by_component: Resources to update versions by theis paths.
        :type version_by_resource_by_component: Dict[ComponentNameStr, Dict[ResourceNameStr, VersionStr]]
        :return: None

        Example:
        >>> UpdateSpec.update_resources(spec, {'banner': {'sandbox_resources.http_server': '2111350684'}})
        """
        logging.info(u'%s.update_resources(spec, `%s`)', cls.__name__, version_by_resource_by_component)
        errors = []
        deploy_units = spec['spec']['deploy_units']
        for component_name, version_by_resource in version_by_resource_by_component.items():
            try:
                deploy_unit_conf = deploy_units[component_name]
                resource_item_by_id = cls._get_resources_items(deploy_unit_conf)
                logging.info(u'Resources for %s: `%s`', component_name, resource_item_by_id)
                for resource, version in version_by_resource.items():
                    try:
                        cls._update_resource_item(resource_item_by_id[resource], version)
                    except Exception as exc:
                        logging.exception('Got exception while updating resource %s/%s', component_name, resource)
                        errors.append(exc)
            except Exception as exc:
                logging.exception('Got exception while updating resource in %s', component_name)
                errors.append(exc)
        if errors:
            raise CompositeError(errors)

    @classmethod
    def update_resources_versions_by_type(cls, spec, version_by_resource_type):
        """
        Update resources by type for all components.
        :param spec: YDeploy config to update.
        :type spec: Dict[str, Any]
        :param version_by_resource_type: Resources types to update to given versions.
        :type version_by_resource_type: Dict[ResourceTypeStr, VersionStr]
        :return: None

        Example:
        >>> UpdateSpec.update_resources_versions_by_type(spec, {'MODADVERT_SM_CONFIG_PROD': '2118728650'})
        """
        logging.info(u'%s.update_resources_versions_by_type(spec, `%s`)', cls.__name__, version_by_resource_type)
        errors = []
        for deploy_unit_name, deploy_unit_conf in spec['spec']['deploy_units'].items():
            try:
                resource_item_by_id = cls._get_resources_items(deploy_unit_conf)
                logging.info(u'Resources for %s: `%s`', deploy_unit_name, resource_item_by_id)
                for resource_type, version in version_by_resource_type.items():
                    if resource_type in resource_item_by_id:
                        logging.info('Found resource type %s in %s', resource_type, deploy_unit_name)
                        cls._update_resource_item(resource_item_by_id[resource_type], version)
                    else:
                        logging.info('Skip %s in %s', resource_type, deploy_unit_name)
            except Exception as exc:
                logging.exception('Got exception while updating resource types in %s', deploy_unit_name)
                errors.append(exc)
        if errors:
            raise CompositeError(errors)

    @classmethod
    def update_description(cls, spec, description):
        if 'revision_info' in spec['spec']:
            spec['spec']['revision_info']['description'] = str(description)
        else:
            logging.warning('No revision_info found')

    @classmethod
    def update_spec(
        cls,
        spec,
        components_with_versions=None,
        version_by_resource_by_component=None,
        version_by_resource_type=None,
        description='',
    ):
        if components_with_versions:
            cls.update_images(spec, components_with_versions)
        if version_by_resource_by_component:
            cls.update_resources(spec, version_by_resource_by_component)
        if version_by_resource_type:
            cls.update_resources_versions_by_type(spec, version_by_resource_type)
        if description:
            cls.update_description(spec, description)


def update_ydeploy_stage(
    stage_name,
    components_with_versions=None,
    version_by_resource_by_component=None,
    version_by_resource_type=None,
    description='',
    popen_caller=subprocess.check_call,
    env=None,
    filename_initial='spec_initial.yaml',
    filename_updated='spec_updated.yaml',
    ya_path='ya',
):
    logging.info('Getting current spec for %s', stage_name)
    popen_caller(
        ['bash', '-c', '{} tool dctl get stage {} > {}'.format(ya_path, stage_name, filename_initial)],
        env=env,
        stdout=subprocess.PIPE,
    )
    with codecs.open(filename_initial, 'r', 'utf-8') as file_initial:
        content = file_initial.read()
        print(content)
        spec = yaml.load(content)
    UpdateSpec.update_spec(
        spec,
        components_with_versions,
        version_by_resource_by_component,
        version_by_resource_type,
        description,
    )
    with codecs.open(filename_updated, 'w', 'utf-8') as file_updated:
        file_updated.write(yaml.dump(spec))
    logging.info('Loading updated spec')
    popen_caller([ya_path, 'tool', 'dctl', 'put', 'stage', filename_updated], env=env)
