# -*- coding: utf-8 -*-
import json
import os
import requests
from sandbox import sdk2

from datetime import datetime
from logging import info, debug
from sandbox.projects.modadvert.common import modadvert
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.environments import PipEnvironment


class State(object):
    def __init__(self, yt_client, path, default=None):
        if default is None:
            default = dict()
        self._yt_client = yt_client
        self._path = path
        self.values = default

    def load(self):
        if self._yt_client.exists(self._path):
            self.values.update(json.loads(''.join(self._yt_client.read_file(self._path))))

    def dump(self):
        parent_node = os.path.dirname(self._path)
        if not self._yt_client.exists(parent_node):
            self._yt_client.create('map_node', parent_node, recursive=True)
        self._yt_client.write_file(self._path, json.dumps(self.values))

    def __enter__(self):
        self.load()
        return self

    def __exit__(self, type, value, traceback):
        self.dump()


class ModadvertRunMailingCircuit(modadvert.ModadvertBaseYtTask):

    class Requirements(sdk2.Task.Requirements):
        environments = (
            PipEnvironment('yandex-yt', '0.8.38a1', use_wheel=True),
            PipEnvironment('yandex-yt-yson-bindings-skynet', use_wheel=True),
        )

    class Parameters(modadvert.ModadvertBaseYtTask.Parameters):
        state_path = sdk2.parameters.String('State path')
        src_dir = sdk2.parameters.String('Yt src directory')
        max_tables_count = sdk2.parameters.Integer('Max tables to process')
        send_to = sdk2.parameters.List('Send to', default=['modadvert-reports@yandex-team.ru'])
        callback_url = sdk2.parameters.String('Callback url', default='http://dev.modadvert-test.yandex-team.ru/api/')

    def get_tables_to_process(self, yt_client, src_table_dir, processed_tables, max_tables_count=None):
        if src_table_dir.endswith('/'):
            src_table_dir = src_table_dir[:-1]

        min_processed_table = min(processed_tables or ['2000-01-01T00:00:00'])
        tables = sorted(yt_client.search(
            src_table_dir,
            node_type='table',
            path_filter=lambda path: (os.path.basename(path) > min_processed_table) and (os.path.basename(path) not in processed_tables)
        ))
        if max_tables_count:
            tables = tables[:max_tables_count]
        return tables

    def on_execute(self):

        # This is a hack for stupid jinja to be able to render a utf-8 template
        # Which it fails to render if it has a russian word
        import sys
        if sys.version_info.major < 3:
            reload(sys)
        sys.setdefaultencoding('utf8')
        import jinja2
        template_path = os.path.dirname(os.path.abspath(__file__))
        jinja_env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_path))
        jinja_template = jinja_env.get_template(os.path.join('report.html'))

        import yt.wrapper
        yt_client = yt.wrapper.client.YtClient(config={
            'proxy': {'url': self.Parameters.yt_proxy_url},
            'token': self.get_yt_token(),
        })

        with State(yt_client, self.Parameters.state_path) as state:
            src_tables = self.get_tables_to_process(
                yt_client,
                self.Parameters.src_dir,
                state.values.get('processed_tables', []),
                self.Parameters.max_tables_count,
            )
            info('Will process tables: {}'.format(src_tables))

            regions_mapping = {159: 'kz', 187: 'ua', 225: 'ru', 149: 'by', 983: 'tr'}

            for src_table in src_tables:
                info('Started processing table %s', src_table)
                rows = list(yt_client.read_table(src_table))
                if rows:
                    self.add_ok_verdicts(rows)
                    info('Table {} has {} rows'.format(src_table, len(rows)))
                    for count, row in enumerate(rows):
                        data = row['data']
                        data = data[0] if isinstance(data, list) else data

                        if (data.get('promo') or {}).get('start_time'):
                            dtime = datetime.fromtimestamp(data['promo']['start_time'] * 0.001)
                            data['promo']['start_time'] = dtime.strftime('%Y.%m.%d %H:%M:%S')

                        if (data.get('promo') or {}).get('stop_time'):
                            dtime = datetime.fromtimestamp(data['promo']['stop_time'] * 0.001)
                            data['promo']['stop_time'] = dtime.strftime('%Y.%m.%d %H:%M:%S')

                        mail_subject = u'Справочник: карточка организации N{} V{}'.format(
                            row.get('meta', {}).get('id'),
                            row.get('meta', {}).get('export_version')
                        )
                        regions = ', '.join(regions_mapping.get(r_id, str(r_id)) for r_id in data.get('geo_targeting', []))
                        message = jinja_template.render(regions=regions, meta=row['meta'], data=data)
                        message = message.replace(u'\u000b', '')
                        debug('Sending mail')
                        debug('Subject: %s', mail_subject)
                        debug('Body: %s', message)
                        channel.sandbox.send_email(
                            mail_to=self.Parameters.send_to,
                            mail_cc=[],
                            mail_subject=mail_subject,
                            mail_body=message,
                            content_type='text/html',
                        )
                else:
                    info('Table %s is empty', src_table)

                info('Finished processing table %s', src_table)
                processed_tables = sorted(state.values.get('processed_tables', []) + [os.path.basename(src_table)])
                state.values.update({'processed_tables': processed_tables[-500:]})
                state.dump()

    def add_ok_verdicts(self, rows):
        verdicts = []
        for row in rows:
            verdicts.append({
                'meta': row['meta'],
                'create_time': row['create_time'],
                'type': row['type'],
                'service': row['service'],
                'verdicts': [{'verdict': 'Yes', 'reasons': []}]
            })

        response = requests.post(
            self.Parameters.callback_url,
            headers={
                'Content-Type': 'application/json',
                'Authorization': 'Oauth {}'.format(self.get_yt_token()),
            },
            data=json.dumps({
                'jsonrpc': '2.0',
                'method': 'add_verdicts',
                'id': 'task_id_{}'.format(self.id),
                'params': {
                    'circuit': 'mailing',
                    'results': verdicts,
                },
            })
        )

        try:
            response.raise_for_status()
        except:
            info('Failed request: {}'.format(vars(response.request)))
            info('add_verdicts error: {}'.format(response.text))
            raise
