# -*- coding: utf-8 -*-
import logging
from enum import Enum

from sandbox import sdk2
from sandbox.common.rest import Client
from sandbox.common.types.misc import NotExists
from sandbox.common.utils import get_resource_link, get_scheduler_link, get_task_link
from sandbox.projects.modadvert.common import modadvert
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.modadvert.common.constants import STARTREK_CLIENT_ENVIROMENTS
from sandbox.projects.modadvert.DeployQloudEnvironment import ModadvertDeployQloudEnvironment
from sandbox.projects.modadvert.DeployYDeployEnvironment import ModadvertDeployYDeployEnvironment


class DeployPolicy(Enum):  # TODO: Remove this after qloud shut down.
    QLOUD = 'QLOUD'
    QLOUD_AND_DEPLOY = 'QLOUD_AND_DEPLOY'
    DEPLOY = 'DEPLOY'


class ModadvertDeployTask(modadvert.ModadvertBaseTask):
    ADDITIONAL_PARAMETERS = []
    COMPONENT = ''
    ENV_NAMES = {}
    RELEASE_COMPONENTS = []
    SCHEDULERS = {}

    class Requirements(modadvert.ModadvertBaseTask.Requirements):
        environments = STARTREK_CLIENT_ENVIROMENTS

    class Parameters(modadvert.ModadvertBaseTask.Parameters):
        vault_user = sdk2.parameters.String(
            'Vault user',
            default=rm_const.COMMON_TOKEN_OWNER,
            required=True
        )
        st_token_vault_name = sdk2.parameters.String(
            'Startrek oauth token vault name',
            default=rm_const.COMMON_TOKEN_NAME,
            required=True
        )

        release_number = sdk2.parameters.Integer('Release number', default=None)
        release_tag_number = sdk2.parameters.String(
            'Tag number. If not specified, the highest existing tag will be used',
            default=None
        )
        deploy_policy = sdk2.parameters.String(  # TODO: Remove this after qloud shut down.
            'One of: {}'.format(list(DeployPolicy.__members__)),
            default=DeployPolicy.QLOUD.value,
        )

    def on_save(self):
        """
        Testenv is badly compatible with SDK2
        it fills Context instead of Parameters
        """

        for parameter_name in [
            'vault_user',
            'st_token_vault_name',
            'release_number',
            'release_tag_number',
            'release_environment',
        ] + self.ADDITIONAL_PARAMETERS:
            context_value = getattr(self.Context, parameter_name)
            if context_value and context_value is not NotExists:
                setattr(self.Parameters, parameter_name, context_value)

        return super(ModadvertDeployTask, self).on_save()

    def get_resource_by_type(self, resource_type):
        if not isinstance(resource_type, list):
            resource_type = [resource_type]
        for param in self.Parameters:
            if type(param[1]) == resource_type or type(param[1]) in resource_type:
                return param[1]

    def on_execute_inner(self):
        deploy_policy = self.Parameters.deploy_policy
        assert deploy_policy in DeployPolicy.__members__, deploy_policy
        env_name = self.ENV_NAMES[self.Parameters.release_environment]

        if self.Parameters.release_number:
            c_info = rmc.COMPONENTS[self.COMPONENT]()

            if self.RELEASE_COMPONENTS:
                release_tag = self.Parameters.release_tag_number or c_info.last_tag_num(self.Parameters.release_number)
                version = '0.{}.{}'.format(self.Parameters.release_number, release_tag)
                components = {component: version for component in self.RELEASE_COMPONENTS}
                commit_message = '{}-{}: Deploy components with version {}'.format(
                    c_info.notify_cfg__st__queue,
                    self.Parameters.release_number,
                    version
                )

                if (
                    deploy_policy in (DeployPolicy.QLOUD.value, DeployPolicy.QLOUD_AND_DEPLOY.value)
                    and not self.Context.deploy_qloud_environment_subtask
                ):
                    self.Context.deploy_qloud_environment_subtask = self.create_subtask(
                        ModadvertDeployQloudEnvironment,
                        dict(
                            vault_user=self.Parameters.vault_user,
                            qloud_token_vault_name='qloud-token',
                            registry_token_vault_name='registry-token',
                            sandbox_token_vault_name='sandbox-token',
                            environment_id='modadvert.{}.{}'.format(self.COMPONENT.lower(), env_name),
                            components=components,
                            resources=self.get_deploying_resources(),
                            commit_message=commit_message,
                        ),
                        description='Deploy Qloud environment for task {}'.format(self.id)
                    )
                if (
                    deploy_policy in (DeployPolicy.DEPLOY.value, DeployPolicy.QLOUD_AND_DEPLOY.value)
                    and not self.Context.deploy_ydeploy_environment_subtask
                ):
                    # TODO: Remove this after qloud shut down.
                    yd_env_name = {'production': 'prod', 'testing': 'test'}.get(env_name, env_name)
                    # TODO: Remove add-verdicts-api from release machine after qloud shut down.
                    yd_components = dict(components)
                    yd_components.pop('add-verdicts-api', None)
                    # Deploy resources either by component and path or by type for all components.
                    resource_version_by_type = self.get_ydeploy_resources_types_new_versions()

                    self.Context.deploy_ydeploy_environment_subtask = self.create_subtask(
                        ModadvertDeployYDeployEnvironment,
                        dict(
                            vault_user=self.Parameters.vault_user,
                            yp_token_vault_name='yp-token',
                            sandbox_token_vault_name='sandbox-token',
                            environment_id='modadvert-{}-{}'.format(self.COMPONENT.lower(), yd_env_name),
                            components=yd_components,
                            resources=[] if resource_version_by_type else self.get_deploying_resources(),
                            resources_versions=resource_version_by_type,
                            commit_message=commit_message,
                        ),
                        description='Deploy ydeploy environment for task {}'.format(self.id)
                    )

            self.wait_all_subtasks()
        else:
            c_info = None

        changes = []
        client = Client()

        for scheduler_name, scheduler_info in self.SCHEDULERS.iteritems():
            scheduler_id = scheduler_info['environments'].get(self.Parameters.release_environment)
            if scheduler_id is None:
                continue
            scheduler = client.scheduler[scheduler_id].read()

            for field in scheduler['task']['custom_fields']:
                if field['name'] in scheduler_info['resources']:
                    resource_type = scheduler_info['resources'][field['name']]
                    resource = self.get_resource_by_type(resource_type)
                    changes.append((scheduler_id, str(resource.type), field['value'], resource.id))
                    field['value'] = resource.id

            scheduler['task']['description'] = (
                '{description} <b>{env}</b>. '
                '<a href="https://st.yandex-team.ru/{ticket}>{ticket}</a>: '
                'Deploy task <a href="{task_link}">{task_id}</a>'
            ).format(
                description=scheduler_info['description'],
                env=self.Parameters.release_environment,
                ticket='{}-{}'.format(c_info.notify_cfg__st__queue if c_info else '', self.Parameters.release_number),
                task_link=get_task_link(self.id),
                task_id=self.id
            )

            client.scheduler[scheduler_id] = scheduler

        changes_message = self.get_deploy_message(changes)
        if c_info:
            try:
                logging.info(
                    'Attempt to get token data %s for owner %s',
                    self.Parameters.st_token_vault_name,
                    self.Parameters.vault_user
                )
                st_token = self.get_author_token(self.Parameters.st_token_vault_name, vault_user=self.Parameters.vault_user)
            except Exception:
                logging.info(
                    'Failed to get token. Attemting to get common token data %s for owner %s',
                    rm_const.COMMON_TOKEN_OWNER,
                    rm_const.COMMON_TOKEN_NAME
                )
                st_token = self.get_author_token(rm_const.COMMON_TOKEN_NAME, rm_const.COMMON_TOKEN_OWNER)

            st_helper = STHelper(st_token)
            st_helper.comment(self.Parameters.release_number, changes_message, c_info)
        else:
            logging.info(changes_message)

    def get_deploy_message(self, changes):
        message_lines = [
            'Deploying **{}** environment:'.format(self.Parameters.release_environment),
            'Task id (({} {}))'.format(get_task_link(self.id), self.id)
        ]
        if changes:
            message_lines.append('#|')
            message_lines.append('**|| Scheduler ID | Resource type | old id | new id ||**')
            message_lines.extend([
                '|| (({} {})) | {} | (({} {})) | (({} {})) ||'.format(
                    get_scheduler_link(scheduler_id),
                    scheduler_id,
                    resource_type,
                    get_resource_link(old_id), old_id,
                    get_resource_link(new_id), new_id
                )
                for (scheduler_id, resource_type, old_id, new_id) in changes
            ])
            message_lines.append('|#')
        return '\n'.join(message_lines)

    def get_deploying_resources(self):
        return []

    def get_ydeploy_resources_types_new_versions(self):
        return {}
