# -*- coding: utf-8 -*-

import logging
import sandbox.sandboxsdk.task as sdk_task
import sandbox.sandboxsdk.parameters as sdk_parameters

from sandbox.sdk2 import yav

from sandbox.projects.modadvert import resource_types


class ModAdvertMonika(sdk_task.SandboxTask):
    """ Monika -> ModAdvert Monitoring """

    type = 'MODADVERT_MONITORING'
    cores = 1
    required_ram = 4096
    execution_space = 10 * 1024

    class BinariesResource(sdk_parameters.LastReleasedResource):
        required = False
        name = 'binaries_resource'
        description = 'Resource with monika binary'
        resource_type = resource_types.MODADVERT_MONIKA_BINARY

    class VaultUser(sdk_parameters.SandboxStringParameter):
        name = 'vault_user'
        description = name
        default_value = 'MODADVERT'

    class YtTokenVaultName(sdk_parameters.SandboxStringParameter):
        name = 'yt_token_vault_name'
        description = name
        default_value = 'yt-token'

    class SolomonSecretName(sdk_parameters.SandboxStringParameter):
        name = 'solomon_token_secret_id'
        description = name

    class MinSuccessesCount(sdk_parameters.SandboxIntegerParameter):
        name = 'min_successes_count'
        description = name
        default_value = 20

    class MaxFailsCount(sdk_parameters.SandboxIntegerParameter):
        name = 'max_fails_count'
        description = name
        default_value = 5

    class SleepSeconds(sdk_parameters.SandboxIntegerParameter):
        name = 'sleep_seconds'
        description = name
        default_value = 5

    input_parameters = [
        BinariesResource,
        VaultUser,
        YtTokenVaultName,
        SolomonSecretName,
        MinSuccessesCount,
        MaxFailsCount,
        SleepSeconds,
    ]

    def get_sandbox_token(self):
        return self.get_vault_data(self.ctx[self.VaultUser.name], 'sandbox-token')

    def get_yql_token(self):
        return self.get_vault_data(self.ctx[self.VaultUser.name], 'yql-token')

    def get_yt_token(self):
        return self.get_vault_data(self.ctx[self.VaultUser.name], self.ctx[self.YtTokenVaultName.name])

    def get_solomon_token(self):
        return yav.Secret(self.ctx[self.SolomonSecretName.name]).data()['SOLOMON_TOKEN']

    def create_env(self):
        return {
            'SANDBOX_TOKEN': self.get_sandbox_token(),
            'YQL_TOKEN': self.get_yql_token(),
            'YT_TOKEN': self.get_yt_token(),
            'SOLOMON_TOKEN': self.get_solomon_token()
        }

    def on_execute(self):
        from projects.modadvert.common.utils import run_command_with_livelog
        logging.info('start Monika on_execute()')

        tarball_path = self.sync_resource(self.ctx[self.BinariesResource.name])
        unpack_binary_cmd = ['tar', '-xvf', tarball_path]
        run_command_with_livelog(unpack_binary_cmd)

        monika_cmd = [
            './monika',
            '--min-successes-count', str(self.ctx[self.MinSuccessesCount.name]),
            '--max-fails-count', str(self.ctx[self.MaxFailsCount.name]),
            '--sleep-seconds', str(self.ctx[self.SleepSeconds.name]),
            '--production',
        ]

        log_message = run_command_with_livelog(monika_cmd, env=self.create_env())
        logging.info(log_message)
        logging.info('finished Monika on_execute()')


__Task__ = ModAdvertMonika
