# -*- coding: utf-8 -*-
import os.path
from sandbox import sdk2
import yaml
import sandbox.projects.common.build.parameters
import sandbox.projects.common.build.YaPackage as YaPackage

from sandbox.common.types.misc import NotExists
from sandbox.projects.modadvert import resource_types
from sandbox.projects.common.apihelpers import get_task_resource_id
from sandbox.projects.modadvert.common import modadvert


class ModadvertRunBinaryWithConfig(modadvert.ModadvertBaseYtTask):
    """
    Run binary with config. (To run binary without config leave config fields blank.)
    """

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(modadvert.ModadvertBaseYtTask.Parameters):

        with sdk2.parameters.Group('Binary') as binary_group:
            binary_resource = sdk2.parameters.Resource(
                'Resource with binary',
                resource_type=resource_types.ALL_MODADVERT_TYPES
            )
            binary_arcadia_url = sdk2.parameters.ArcadiaUrl('Arcadia url for binary build')
            binary_arcadia_patch = sdk2.parameters.String('A patch for binary build')
            binary_package_config = sdk2.parameters.String('A path to binary package config')
            binary_dir = sdk2.parameters.String('A base dir for binary file')

        with sdk2.parameters.Group('Config') as config_group:
            config_resource = sdk2.parameters.Resource(
                'Resource with config',
                resource_type=resource_types.ALL_MODADVERT_TYPES
            )
            config_arcadia_url = sdk2.parameters.ArcadiaUrl('Arcadia url for config build')
            config_arcadia_patch = sdk2.parameters.String('A patch for config build')
            config_package_config = sdk2.parameters.String('A path to config package config')
            config_dir = sdk2.parameters.String('A base dir for config file')

        with sdk2.parameters.Group('Launch parameters') as launch_group:
            base_cmd = sdk2.parameters.List('Base cmd', default=['./binary'], required=True)
            cmd_options = sdk2.parameters.Dict('Binary options', default={'--config': './config.yaml'})
            debug = sdk2.parameters.Bool('Debug mode', default=True, required=True)
            configs_patch = sdk2.parameters.String('Yaml configs patch', multiline=True)

    def create_command(self):
        cmd = self.Parameters.base_cmd
        for pair in self.Parameters.cmd_options.iteritems():
            cmd.extend(pair)

        if self.Parameters.debug and self.Parameters.debug is not NotExists:
            cmd.append('--debug')

        return cmd

    @staticmethod
    def get_default_subtask_parameters():
        return {
            YaPackage.PackageTypeParameter.name: YaPackage.TARBALL,
            sandbox.projects.common.build.parameters.BuildType.name: YaPackage.RELEASE,
            YaPackage.CompressPackageArchiveParameter.name: True,
            YaPackage.UseNewFormatParameter.name: True,
            YaPackage.PublishPackageParameter.name: False,
        }

    def on_before_execute(self):
        super(ModadvertRunBinaryWithConfig, self).on_before_execute()

        if not self.Parameters.binary_resource and not self.Context.build_binary_subtask:
            self.Context.build_binary_subtask = self.create_subtask(
                'YA_PACKAGE',
                {
                    sandbox.projects.common.build.parameters.ArcadiaUrl.name: self.Parameters.binary_arcadia_url,
                    sandbox.projects.common.build.parameters.ArcadiaPatch.name: self.Parameters.binary_arcadia_patch,
                    YaPackage.PackagesParameter.name: self.Parameters.binary_package_config,
                    YaPackage.ResourceTypeParameter.name: type(self.Parameters).binary_resource.resource_type,
                },
                description='Building binary for task {}'.format(self.id)
            )

        if self.Parameters.config_package_config and self.Parameters.config_package_config is not NotExists:
            if not self.Parameters.config_resource and not self.Context.build_config_subtask:
                self.Context.build_config_subtask = self.create_subtask(
                    'YA_PACKAGE',
                    {
                        sandbox.projects.common.build.parameters.ArcadiaUrl.name: self.Parameters.config_arcadia_url,
                        sandbox.projects.common.build.parameters.ArcadiaPatch.name: self.Parameters.config_arcadia_patch,
                        YaPackage.PackagesParameter.name: self.Parameters.config_package_config,
                        YaPackage.ResourceTypeParameter.name: type(self.Parameters).config_resource.resource_type,
                    },
                    description='Building config for task {}'.format(self.id),
                )

        self.wait_all_subtasks()

        if not self.Parameters.binary_resource:
            self.Parameters.binary_resource = get_task_resource_id(
                self.Context.build_binary_subtask,
                resource_types.YA_PACKAGE.name
            )

        if self.Parameters.config_package_config and self.Parameters.config_package_config is not NotExists and not self.Parameters.config_resource:
            self.Parameters.config_resource = get_task_resource_id(
                self.Context.build_config_subtask,
                resource_types.YA_PACKAGE.name
            )

    @staticmethod
    def override_config_file(config_file, config_patch):
        overriden_config = {}
        if os.path.isfile(config_file):
            with open(config_file, 'r') as stream:
                overriden_config = yaml.load(stream)
        with open(config_file, 'w') as stream:
            yaml.dump(modadvert.merge_configs(config_patch, overriden_config), stream)

    def on_execute_inner(self):
        binary_dir = self.Parameters.binary_dir
        self.untar_resource(self.Parameters.binary_resource, binary_dir if binary_dir and binary_dir is not NotExists else None)

        if self.Parameters.config_resource and self.Parameters.config_resource is not NotExists:
            config_dir = self.Parameters.config_dir
            self.untar_resource(self.Parameters.config_resource, config_dir if config_dir and config_dir is not NotExists else None)

        if self.Parameters.configs_patch:
            for config_file, config_patch in yaml.load(self.Parameters.configs_patch).iteritems():
                self.override_config_file(config_file, config_patch)

        self.run_command(self.create_command(), log_prefix='main', env=self.create_env())
