# -*- coding: utf-8 -*-
import logging
from sandbox import sdk2
import sandbox.common.types.client as ctc

from sandbox.projects.modadvert.common import modadvert
from sandbox.projects.modadvert import resource_types


class ModAdvertRunBroker(modadvert.ModadvertBaseRunSupermoderation):
    name = "MODADVERT_RUN_SM_BROKER"

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC

    class Parameters(modadvert.ModadvertBaseRunSupermoderation.Parameters):
        with sdk2.parameters.Group('Binary') as binary_group:
            binaries_resource = sdk2.parameters.Resource(
                'Resource with broker binary',
                resource_type=[resource_types.MODADVERT_SM_BROKER_BINARY],
                required=True
            )

        with modadvert.ModadvertBaseRunSupermoderation.Parameters.launch_group() as launch_group:
            workers_count = sdk2.parameters.Integer('Workers count', required=False, default=5)
            readers_count = sdk2.parameters.Integer('Readers count', required=False, default=1)
            chunk_size = sdk2.parameters.Integer('Chunk size', required=False)
            duplicate_lb_to_yt = sdk2.parameters.Bool('Duplicate LB to YT', default=False, required=False)
            turn_off_lb = sdk2.parameters.Bool('Turn off LB', default=False, required=False)
            turn_off_response_push = sdk2.parameters.Bool('Turn off response push', default=False, required=False)

    def create_command(self):
        cmd = super(ModAdvertRunBroker, self).create_command()
        cmd.extend(['--workers-count', self.Parameters.workers_count])
        if self.Parameters.readers_count > 1:
            cmd.extend(['--partition-count', self.Parameters.readers_count])
        if self.Parameters.duplicate_lb_to_yt:
            cmd.append('--duplicate-lb-to-yt')

        if self.Parameters.turn_off_lb:
            cmd.append('--turn-off-lb')

        if self.Parameters.turn_off_response_push:
            cmd.append('--turn-off-response-push')

        if self.Parameters.chunk_size:
            cmd.extend(['--chunk-size', self.Parameters.chunk_size])

        return cmd

    def on_execute_inner(self):
        base_command = self.create_command()
        env = self.create_env()
        job_count = self.Parameters.readers_count
        jobs = []
        for job_num in xrange(job_count):
            log_prefix = 'main-{}'.format(job_num)
            pl = sdk2.helpers.ProcessLog(self, logger=log_prefix).__enter__()
            additional_args = ['--partition-num', job_num]
            command = map(str, base_command + additional_args)
            logging.info('RUNNING COMMAND: {} JOB NUM {}'.format(command, job_num))
            job = sdk2.helpers.subprocess.Popen(command, stdout=pl.stdout, stderr=pl.stdout, env=env)
            jobs.append((job, pl))

        retcodes = []
        for job_num, (job, pl) in enumerate(jobs):
            retcodes.append(job.wait())
            pl.logger.info('Job {} finished'.format(job_num))
            pl.__exit__(None, None, None)

        for job_num, retcode in enumerate(retcodes):
            if retcode != 0:
                raise Exception(
                    'Error in job {} return code={}'.format(job_num, retcode)
                )
