from collections import defaultdict
from itertools import groupby, chain
from operator import itemgetter
from six.moves import map

VERDICTS = ['ACCEPT', 'NA', 'MANUAL', 'REJECT', None]
ITEM_TEMPLATE = '{feature} !!(green)+{add}!! !!(red)-{remove}!! (!!({color}){diff_perc:+.2%}!!)'
TOTAL_ITEM_TEMPLATE = '!!(gray){diff}!! / {count} ({diff_perc:.2%})'


def get_verdicts_stat(rows):
    """
    Calculate stats over verdicts for single case.
    For each verdict return counts of feature, base, added, removed objects and diff_perc.
    Also calculate total count of objects and count of objects with switched verdicts.
    """
    verdicts_stat = defaultdict(lambda: {'feature': 0, 'base': 0, 'add': 0, 'remove': 0})
    total_count = 0
    total_diff = 0

    for row in rows:
        count = row['count']
        verdicts_stat[row['feature_verdict']]['feature'] += count
        verdicts_stat[row['base_verdict']]['base'] += count
        total_count += count
        if row['feature_verdict'] != row['base_verdict']:
            verdicts_stat[row['feature_verdict']]['add'] += count
            verdicts_stat[row['base_verdict']]['remove'] += count
            total_diff += count

    for verdict, stat in verdicts_stat.iteritems():
        diff = float(stat['add'] - stat['remove'])
        if stat['base']:
            stat['diff_perc'] = diff / stat['base']
        else:
            stat['diff_perc'] = float('inf') * diff if diff else 0.

    verdicts_stat['total'] = {
        'count': total_count,
        'diff': total_diff,
        'diff_perc': float(total_diff) / total_count
    }
    return verdicts_stat


def get_message_item(stat, tolerance=0.05):
    if not stat:
        return ''
    diff_perc = stat['diff_perc']
    return ITEM_TEMPLATE.format(
        color='gray' if abs(diff_perc) < tolerance else 'green' if diff_perc > 0. else 'red',
        **stat
    )


def get_stat_table(key_fields, sorted_rows, highlight_tolerance=0.05, hide_tolerance=-1):
    u"""
    Format stat message for lyncher b2b

    #|
    || type | NA | REJECT | total ||
    || a | 170 !!(green)+100!! !!(red)-30!! (!!(gray)+70.00%!!) | 30 !!(green)+30!! !!(red)-100!! (!!(gray)-70.00%!!) | !!(gray)130!! / 200 (65.00%) ||
    **|| b | 0 !!(green)+0!! !!(red)-100!! (!!(red)-100.00%!!) | 100 !!(green)+100!! !!(red)-0!! (!!(green)+inf%!!) | !!(gray)100!! / 100 (100.00%) ||**
    |#"""
    assert hide_tolerance < highlight_tolerance

    stat_list = []
    for keys, rows in groupby(sorted_rows, itemgetter(*key_fields)):
        stat_list.append((list(keys) if isinstance(keys, tuple) else [keys], get_verdicts_stat(rows)))

    message_lines = []
    def add_message_line(*line_items):
        message_lines.append('|| {} ||'.format(' | '.join(map(str, chain.from_iterable(line_items)))))

    message_lines.append('#|')
    add_message_line(key_fields, VERDICTS, ['total'])  # add table header
    for keys, verdicts_stat in stat_list:
        if verdicts_stat['total']['diff_perc'] > hide_tolerance:
            add_message_line(
                keys,
                (get_message_item(verdicts_stat.get(verdict), highlight_tolerance) for verdict in VERDICTS),
                [TOTAL_ITEM_TEMPLATE.format(**verdicts_stat['total'])]
            )
            if verdicts_stat['total']['diff_perc'] > highlight_tolerance:
                message_lines[-1] = '**{}**'.format(message_lines[-1])
    message_lines.append('|#')

    return '\n'.join(message_lines)
