import logging
import urlparse
import requests

from enum import Enum

import sandbox.common.types.client as ctc

from sandbox import common
from sandbox import sdk2
from sandbox.common.types import notification as ctn
from sandbox.common.types import task as ctt
from sandbox.common.types.misc import NotExists
from sandbox.projects.common import decorators
from sandbox.sandboxsdk.environments import PipEnvironment

SERVER = 'teamcity.browser.yandex-team.ru'
MEASURE_BUILD_TYPE = 'Browser_Tests_Perf_Measure_MeasureNoWait'
TAGS = ['performace_tests_check']
RESULTS_PATH = 'web_loading/full.results.ya.json'
PRODUCT = 'web-direct-banners'
POOL = 'browser zombie win7-fast'


class TeamcityResults(sdk2.Resource):
    ttl = 7


class BuildWaiter(object):

    class Status(Enum):
        QUEUED = 'status_wait_time'
        RUNNING = 'status_executing'
        SUCCESS = 'status_success'
        FAILURE = 'status_exception'
        UNKNOWN = 'status_draft'
        UNKNOWN_FINISHED = 'status_exception'

    def __init__(self, tc_client, build):
        self.client = tc_client
        self.build = build
        self.status = None

    @decorators.retries(20, delay=0.5, backoff=2, max_delay=512, exceptions=(ValueError,))
    def update_status(self):
        if not self.finished():
            if self.build.state == 'queued':
                self.status = BuildWaiter.Status.QUEUED
            elif self.build.state == 'running':
                self.status = BuildWaiter.Status.RUNNING
            elif self.build.state == 'finished':
                if self.build.status == 'SUCCESS':
                    self.status = BuildWaiter.Status.SUCCESS
                elif self.build.status == 'FAILURE':
                    self.status = BuildWaiter.Status.FAILURE
                elif self.build.status == 'UNKNOWN':
                    self.status = BuildWaiter.Status.UNKNOWN_FINISHED
                else:
                    logging.warning(
                        'Unknown finished build status for {}: {}'.format(self.build.web_url, self.build.status)
                    )
                    self.status = BuildWaiter.Status.UNKNOWN_FINISHED
            else:
                logging.warning(
                    'Unknown build state for {}: {}'.format(self.web_url, build.state))
                self.status = BuildWaiter.Status.UNKNOWN

    def finished(self):
        return self.status in (
            BuildWaiter.Status.SUCCESS,
            BuildWaiter.Status.FAILURE,
            BuildWaiter.Status.UNKNOWN_FINISHED
        )

    @decorators.retries(5, delay=1, backoff=1, max_delay=1, exceptions=(ValueError,))
    def download_results(self, path, dst):
        self.build.download_artifact(path=path, dst=dst)


class ModadvertWaitTeamcityBuild(sdk2.Task):

    waiter = None

    class Requirements(sdk2.Requirements):
        disk_space = 100  # MB
        client_tags = ctc.Tag.BROWSER  # because of teamcity access
        cores = 1
        environments = [
            PipEnvironment('teamcity-client', '4.8.2'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        kill_timeout = 3 * 60 * 60

        notifications = (
            sdk2.Notification(
                (ctt.Status.FAILURE, ctt.Status.EXCEPTION, ctt.Status.NO_RES, ctt.Status.TIMEOUT, ctt.Status.STOPPED),
                ('st-mks',),
                ctn.Transport.EMAIL
            ),
        )

        resource_id = sdk2.parameters.String('Id of SB resource with WPR archive')

        tc_build_id = sdk2.parameters.String('Teamcity build id (if already started)')

        sleep_time = sdk2.parameters.Integer('Sleep time (minutes)', default=3)

        with sdk2.parameters.Group('Credentials') as credentials_group:
            browser_teamcity_vault = sdk2.parameters.String('Vault item with token for browser teamcity')

    class Context(sdk2.Context):
        build_id = None

    def trigger_build(self, tc_client):
        properties = {
            'product': PRODUCT,
            'spec': self.Parameters.resource_id,
        }

        build = tc_client.BuildType(id=MEASURE_BUILD_TYPE).run_build(
            pool=tc_client.Pool(name=POOL),
            properties=properties,
            tags=TAGS
        )

        logging.info('Build started. Build url: {}'.format(build.web_url))
        return build

    @decorators.retries(5, delay=1, backoff=1, max_delay=1, exceptions=(ValueError,))
    def get_teamcity_client(self):
        import teamcity_client.client
        return teamcity_client.client.TeamcityClient(
            server_url=SERVER,
            auth=sdk2.Vault.data(self.Parameters.browser_teamcity_vault)
        )

    def load_build(self, tc_client):
        if self.Context.build_id is None:
            build = self.trigger_build(tc_client)
            self.Context.build_id = build.id
            return build
        else:
            return tc_client.Build(id=self.Context.build_id)

    def on_execute(self):
        resource_id_not_exists = self.Parameters.resource_id is NotExists or not self.Parameters.resource_id
        tc_build_id_not_exists = self.Parameters.tc_build_id is NotExists or not self.Parameters.tc_build_id
        if resource_id_not_exists and tc_build_id_not_exists:
            raise ValueError('Either resource id or build id should be provided!')

        if self.Parameters.tc_build_id:  # If build id is provided, wait for that build
            self.Context.build_id = self.Parameters.tc_build_id

        tc_client = self.get_teamcity_client()

        build = self.load_build(tc_client)
        self.waiter = BuildWaiter(tc_client, build)

        self.waiter.update_status()

        if not self.waiter.finished():
            raise sdk2.WaitTime(self.Parameters.sleep_time * 60)

        self.waiter.download_results(
            path=RESULTS_PATH,
            dst='results.json'
        )

        resource = TeamcityResults(self, 'Teamcity json results', 'results.json')
        sdk2.ResourceData(resource).ready()

    @sdk2.header()
    def header(self):
        rows = []

        if self.waiter and self.waiter.status:
            rows.append({
                'Status': '<span class=\'status {}\'>{}</span>'.format(
                    self.waiter.status[1].value, self.waiter.status[0].name),
                'URL': '<a href=\'{0}\'>{0}</a>'.format(self.waiter.build.veb_url())
            })

        return [{
            'content': {'': rows},
            'helperName': '',
        }]
