# -*- coding: utf-8 -*-

import os
import logging

import sandbox.common.types.client as ctc

from sandbox.projects.common import constants as consts
from sandbox.projects.common.build import parameters as ya_make_parameters
from sandbox.projects.common.build.YaMake import YaMakeTask
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.utils import svn_last_change
from sandbox import common

from sandbox.projects.morda_blender import MORDA_BLENDER_BUNDLE
from sandbox.sandboxsdk.parameters import SandboxRadioParameter

logger = logging.getLogger(__name__)


class ReleaseTypeParam(SandboxRadioParameter):
    name = 'release_type'
    description = 'Release type'
    choices = [
        ('production', 'production'),
        ('testing', 'testing')
    ]
    required = True
    group = 'Release type'


class MordaBlender(nanny.ReleaseToNannyTask, YaMakeTask):
    """\
        Build Morda Blender Resources
    """

    type = 'BUILD_MORDA_BLENDER'

    input_parameters = [
        ya_make_parameters.ArcadiaUrl,
        ya_make_parameters.UseArcadiaApiFuse,
        ya_make_parameters.ClearBuild,
        ya_make_parameters.StripBinaries,
        ya_make_parameters.TestParameter,
        ya_make_parameters.TestCustomParameters,
        ya_make_parameters.Sanitize,
        ya_make_parameters.DefinitionFlags,
        ya_make_parameters.CheckReturnCode,
        ya_make_parameters.KeepOn,
        ya_make_parameters.YaTimeout,
        ReleaseTypeParam
    ]

    client_tags = ctc.Tag.Group.LINUX
    execution_space = 90 * 1024  # 90Gb will be enough
    required_ram = 4 * 1024
    cores = 24

    target_path = 'yweb/morda_blender'
    rev_attr = 'rev'

    arts = []
    arts_source = []

    def initCtx(self):
        YaMakeTask.initCtx(self)
        self.ctx['result_single_file'] = False
        self.ctx[consts.BUILD_BUNDLE_KEY] = False
        self.ctx['result_rd'] = 'MordaBlender Bundle'
        logger.info('context: %s', self.ctx)

    def get_build_type(self):
        return consts.RELEASE_BUILD_TYPE

    def get_resources(self):
        return {
            MORDA_BLENDER_BUNDLE.name: {
                'description': MORDA_BLENDER_BUNDLE.name,
                'resource_type': MORDA_BLENDER_BUNDLE,
                'resource_path': 'dst'
            }
        }

    def get_targets(self):
        return [self.target_path]

    def pre_build(self, source_dir):
        self.arts = [{'path': '{}/bin/morda_blender'.format(self.target_path),
                      'dest': 'dst'},
                     {'path': '{}/formulas'.format(self.target_path),
                      'dest': 'dst'}]
        self.arts_source = [{'path': '{}/configs/{}_config.json'.format(self.target_path, self.ctx.get(ReleaseTypeParam.name)),
                             'dest': 'dst/config'},
                            {'path': '{}/scripts/push_client_log_parsing.py'.format(self.target_path),
                             'dest': 'dst/scripts'}]

        for formula_dir in ('formulas', 'bandit_formulas'):
            for domain in ['morda_touch', 'search_app', 'desktop', 'browser']:
                for type in ['blocks', 'verticals', 'navpanel', 'services_tabs']:
                    orig_dir = '{}/{}/{}/{}/{}'.format(source_dir, self.target_path, formula_dir, domain, type)
                    src_dir = '{}/{}/{}/{}'.format(self.target_path, formula_dir, domain, type)
                    if not os.path.exists(orig_dir):
                        continue

                    for factors_file in os.listdir(orig_dir):
                        if factors_file.endswith('.factors'):
                            formula = '{}/{}'.format(src_dir, factors_file)
                            logger.debug('add {}'.format(formula))
                            self.arts_source.append({'path': formula,
                                                    'dest': 'dst/{}/{}/{}'.format(formula_dir, domain, type)})

    def get_arts(self):
        return self.arts

    def get_arts_source(self):
        return self.arts_source

    def get_resources_attrs(self):
        if self.rev_attr in self.ctx:
            release_params = self.get_release_attributes()
            return {MORDA_BLENDER_BUNDLE.name: release_params}
        return {}

    def on_execute(self):
        # use revision from URL or get last from SVN
        parts = self.ctx[consts.ARCADIA_URL_KEY].split('@')
        if len(parts) == 2:
            self.ctx[self.rev_attr] = parts[1]
        else:
            self.set_rev(self.get_last_rev())
        logger.info('on_execute(), context = {}'.format(self.ctx))
        return YaMakeTask.on_execute(self)

    def on_release(self, additional_parameters):
        is_prod = additional_parameters['release_status'] in ('stable', 'prestable')
        sanitize = self.ctx[ya_make_parameters.Sanitize.name]

        if is_prod and sanitize:
            message = 'Failed to release task: tried to deploy sanitizer binaries to production!'
            self.set_info(message)
            raise common.errors.ReleaseError(message)

        if is_prod and self.ctx[ReleaseTypeParam.name] != 'production':
            message = 'Failed to release task: tried to deploy testing configs to production!'
            self.set_info(message)
            raise common.errors.ReleaseError(message)

        logger.info('on_release(), context = {}, additional_parameters = {}'.format(self.ctx, additional_parameters))
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        YaMakeTask.on_release(self, additional_parameters)

    def get_last_rev(self):
        if self.rev_attr in self.ctx:
            return self.ctx[self.rev_attr]
        url = '{}/{}'.format(self.ctx[consts.ARCADIA_URL_KEY], self.target_path)
        logger.info('get revision for {}'.format(url))
        return svn_last_change(url)

    def set_rev(self, rev):
        if self.rev_attr not in self.ctx:
            self.ctx[self.rev_attr] = rev
            self.ctx[consts.ARCADIA_URL_KEY] += '@' + rev

    def get_release_attributes(self, rev=None, rel=None, san=None):
        return {
            self.rev_attr: rev or self.ctx[self.rev_attr],
            'rel': rel or self.ctx[ReleaseTypeParam.name],
            'san': san or self.ctx[ya_make_parameters.Sanitize.name]
        }
