import logging

import sandbox.common.types.task as ctt
from sandbox.projects.common import constants as consts
from sandbox.projects.common.apihelpers import get_last_resource_with_attrs
from sandbox.projects.morda_blender import MORDA_BLENDER_BUNDLE
from sandbox.projects.morda_blender.MordaBlender import MordaBlender
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import sandboxapi
from sandbox.sandboxsdk.channel import channel

logger = logging.getLogger(__name__)


class MordaBlenderCi(MordaBlender):
    """\
        MordaBlender continuous integration
    """

    type = 'BUILD_MORDA_BLENDER_CI'

    execution_space = 10 * 1024  # 10Gb will be enough

    subtask_attr = 'subtask'

    def on_execute(self):
        logger.info('on_execute: context = {}'.format(self.ctx))
        if self.subtask_attr in self.ctx:
            subtask_id = self.ctx[self.subtask_attr]
            logger.info('subtask = {}'.format(subtask_id))
            subtask = channel.sandbox.get_task(subtask_id)

            if subtask.status in (ctt.Status.FAILURE, ctt.Status.EXCEPTION):
                raise errors.SandboxTaskFailureError('BUILD_MORDA_BLENDER subtask failed!')

            logger.info('create_release for {}'.format(subtask))
            rel_id = self.create_release(subtask.id, sandboxapi.RELEASE_TESTING, subject=subtask.description)
            if rel_id is None:
                raise errors.SandboxTaskFailureError('Release failed!')
            else:
                logger.info('Released: {}'.format(rel_id))
        else:
            logger.info('no subtask found, create new one if required')
            rev = self.get_last_rev()
            release_attributes = self.get_release_attributes(rev=rev)  # take rel & san from context
            logger.info('looking for resource MORDA_BLENDER_BUNDLE with attributes \'{}\''.format(release_attributes))
            res = get_last_resource_with_attrs(MORDA_BLENDER_BUNDLE, attrs=release_attributes, all_attrs=True)
            if res is None:
                logger.info('resource not found, create subtask')
                self.set_rev(rev)
                logger.info('create subtask with repo {}'.format(self.ctx[consts.ARCADIA_URL_KEY]))
                subtask = self.create_subtask(
                    task_type=MordaBlender.type,
                    priority=('SERVICE', 'HIGH'),
                    description='MordaBlender CI @{}'.format(rev)
                )
                logger.info('created subtask {} with context {}'.format(subtask, subtask.ctx))
                self.ctx[self.subtask_attr] = subtask.id
                self.wait_tasks(subtask.id, list(ctt.Status.Group.BREAK + ctt.Status.Group.FINISH), wait_all=True)
            else:
                logger.info('resource found {} for attributes \'{}\' '.format(res.id, release_attributes))

    def on_release(self, additional_parameters):
        pass

    def get_resources(self):
        return {}

    def get_targets(self):
        return []

    def get_arts(self):
        return []

    def get_arts_source(self):
        return []

    def get_resources_attrs(self):
        return {}
