import logging
import json
from sandbox import common

from sandbox.projects.msearch.common.context import ReclusterCtx


class ReclusterPreparePhase1(ReclusterCtx):

    def __init__(self, oauth_token=False, rewrite_recluster=False, dryrun=True):

        super(ReclusterPreparePhase1, self).__init__()
        logging.info("ReclusterPreparePhase1 prepare initialization")
        self.oauth_token = oauth_token
        self.dryrun = dryrun
        self.rewrite_recluster = rewrite_recluster

    def run(self):

        # Phase 1. Check that recluster can be continue and generate recluster searchmap.
        logging.info("Recluster Phase 1 is starting.")
        mage_last_rev_status = self.get_mage_last_revision(self.mage_url)

        if not mage_last_rev_status:
            raise common.errors.TaskFailure("Mage ruchkas return error. Exiting. "
                                            "Details: {0}".format(str(mage_last_rev_status)))
        logging.info('Mage get last rev: {0}'.format(str(mage_last_rev_status)))

        if mage_last_rev_status['status'] == 'recluster_activating':
            raise common.errors.TaskFailure("Previous reclusterization pending. Wait.")

        if not self.check_nanny_active(self.nanny_groups):
            raise common.errors.TaskFailure("Cannot continue. "
                                            "Last revision in groups: {0} must be in ACTIVE or ACTIVATING state."
                                            .format(self.nanny_groups))

        if mage_last_rev_status['status'] == 'started':
            logging.info("Previous reclusterization in 'started' state. "
                         "We need start another recluster with same revison.")

            # Use prev revision as gencfg last rev, exception here if mage_last_rev_status returns False
            gencfg_last_rev = mage_last_rev_status['revision']

            # If we already have recluster into mage ruchkas - go to ph2.
            logging.info("Phase 1 finished because lightweight recluster selected for revision: {0}"
                         .format(gencfg_last_rev))
            return gencfg_last_rev

        gencfg_last_rev = self.get_gencfg_last_revision(self.gencfg_url)
        logging.info('Gencfg get last rev: {0}'.format(str(gencfg_last_rev)))

        logging.info('Trying to get nanny gencfg releases. For groups: {0}'.format(str(self.nanny_groups)))
        gencfg_nanny_releases = self.get_gencfg_releases(self.nanny_groups)
        nanny_uniq_release = self.check_gencfg_nanny_releases(gencfg_nanny_releases)

        if not nanny_uniq_release:
            raise common.errors.TaskFailure("Dup. nanny rev found. "
                                            "Aborting. "
                                            "Recluster in progress? {0}".format(gencfg_nanny_releases))

        if not self.revisions_comparsion(nanny_uniq_release, gencfg_last_rev):
            raise common.errors.TaskFailure(
                "Nanny uniq revision: {0} is equal or "
                "bigger than last gencfg revision: {1}. "
                "Aborting.".format(nanny_uniq_release,
                                   gencfg_last_rev))
        else:
            logging.info("Nanny uniq revision: {0} is smaller than last gencfg revision: {1}. "
                         "Continue.".format(nanny_uniq_release,
                                            gencfg_last_rev))

        logging.info("Nanny uniq release: {0}".format(nanny_uniq_release))

        if self.rewrite_recluster:
            recluster_url = \
                "{0}{1}/genmaprec?revision=tags/{2}&prev_revision=tags/{3}&force=yes_i_want_this_uguu".format(
                    self.mage_url,
                    self.project,
                    gencfg_last_rev,
                    nanny_uniq_release)
        else:
            recluster_url = "{0}{1}/genmaprec?revision=tags/{2}&prev_revision=tags/{3}".format(
                self.mage_url,
                self.project,
                gencfg_last_rev,
                nanny_uniq_release
            )

        recluster_steps_ph1 = [{"comment": "Recluster", "url": recluster_url, "oauth_token": ""}]

        try:
            for step in recluster_steps_ph1:
                logging.info("Starting step:{0}".format(step['comment']))
                oauth_token = None

                if step.get("oauth_token", False):
                    oauth_token = step.get("oauth_token")

                if not self.dryrun:
                    logging.info("Url: {0}, Action: get Comment: {1}".format(step["url"], step["comment"]))
                    ret = self.make_auth_req(step["url"], oauth_token, timeout=900)
                else:
                    logging.info("Dryrun fake. Url: {0}, Action: get Comment: {1}".format(step["url"], step["comment"]))
                    ret = True

                if not ret:
                    raise common.errors.TaskFailure("ERROR! Can't generate recluster map through MAGE")
                logging.info("Step:{0} completed with message:{1}".format(step['comment'],
                                                                          ret))
        except Exception as e:
            raise common.errors.TaskFailure("Cannot recluster Phase 1 hosts. "
                                            "Aborting. "
                                            "Error:{0}".format(e))

        if not self.dryrun:
            # Setting that - we already have searchmap, dont generate recluster in second time, if any.
            logging.info("Phase 1 setting mage_last revision to 'started'")
            mage_set_last_rev_status = self.set_mage_last_revision(self.mage_url,
                                                                   gencfg_last_rev,
                                                                   'started')
        else:
            logging.info("Dryrun fake. Phase 1 setting mage_last revision to 'started'")
            mage_set_last_rev_status = "dryrun"

        if not mage_set_last_rev_status:
            raise common.errors.TaskFailure("Cannot set mage rev status. "
                                            "Exiting. "
                                            "Details:{0}".format(mage_set_last_rev_status))

        logging.info("Recluster Phase 1 is finished. Recluster maps is ready")
        return gencfg_last_rev


class ReclusterPreparePhase2(ReclusterCtx):

    def __init__(self, oauth_token=False, gencfg_last_rev=False, dryrun=True):
        logging.info("ReclusterPreparePhase2 prepare initialization")
        super(ReclusterPreparePhase2, self).__init__()
        self.oauth_token = oauth_token
        self.gencfg_last_rev = gencfg_last_rev
        self.dryrun = dryrun

    def run(self):
        # Phase 2. Update self.revision_filename and gencfg groups in services to latest revision.

        logging.info("Recluster Phase 2 is starting.")

        if not self.gencfg_last_rev:
            raise common.errors.TaskFailure("Gencfg last revision for recluster is not specified. "
                                            "Aborting.")

        update_nanny_files_url = "{nanny_url}services/{servicename}/runtime_attrs/resources/files/static_files/"

        # If we start recluster and we interrupted - we cannot continue in future runs.
        if not self.dryrun:
            logging.info("Phase 2 Setting mage_last revision to 'recluster_activating'")
            mage_set_last_rev_status = self.set_mage_last_revision(self.mage_url,
                                                                   self.gencfg_last_rev,
                                                                   'recluster_activating')
        else:
            logging.info("Dryrun fake. Phase 2 setting mage_last revision to 'recluster_activating'")
            mage_set_last_rev_status = "dryrun"

        if not mage_set_last_rev_status:
            raise common.errors.TaskFailure("Cannot set mage rev status. "
                                            "Exiting. "
                                            "Details:{0}".format(mage_set_last_rev_status))

        recluster_rev = "recluster_{0}:tags/{1}".format(self.searchmap_prefix,
                                                        self.gencfg_last_rev)
        logging.info("Recluster revision is: {0}".format(recluster_rev))

        recluster_steps_ph2 = [{
            "comment": "Sandbox auto task update {revision_filename}".format(
                revision_filename=self.revision_filename
            ),
            "url": update_nanny_files_url,
            "meta": {},
            "oauth_token": self.oauth_token,
            "put": {
                "content": {
                    "local_path": self.revision_filename,
                    "content": recluster_rev,
                    "is_dynamic": True,
                    "extract_path": ""
                },
                "comment": "Sandbox auto recluster task. Update {revision_filename} to: {recluster_revison}".format(
                    recluster_revison=recluster_rev,
                    revision_filename=self.revision_filename),
                "meta_info": {"is_disposable": False}
            }
        }]

        # Update self.revision_filename

        try:
            for service in self.nanny_groups:
                for step in recluster_steps_ph2:
                    logging.info("Starting step:{0}".format(step['comment']))
                    oauth_token = None

                    if step.get("oauth_token", False):
                        oauth_token = step.get("oauth_token")

                    if not self.dryrun:
                        logging.info("Url: {0}, Action: put, Content: {1}".format(step["url"]
                                                                                  .format(nanny_url=self.nanny_url,
                                                                                          servicename=service),
                                                                                  step.get("put", False)))
                        ret = self.make_auth_req(step["url"].format(nanny_url=self.nanny_url,
                                                                    servicename=service),
                                                 token=oauth_token,
                                                 put=step.get("put", False),
                                                 timeout=900)
                    else:
                        logging.info("Dryrun fake. Url: {0}, Action: put, Content: {1}"
                                     .format(step["url"]
                                             .format(nanny_url=self.nanny_url,
                                                     servicename=service),
                                             step.get("put", False)))
                        ret = True

                    if not ret:
                        raise common.errors.TaskFailure("Nanny request failed. "
                                                        "Update file to recluster rev failed.")

                    logging.info("Step:{0} completed.".format(step['comment']))

        except Exception as e:
            raise common.errors.TaskFailure("Cannot recluster hosts. Aborting. Error:{0}".format(e))

        # Add gencfg release to all gencfg groups

        gencfg_service_groups = self.get_gencfg_releases_full(self.nanny_groups)
        update_nanny_instances_url = "{nanny_url}services/{servicename}/runtime_attrs/instances/"

        # So  we need update group with reclsuter revision, for all services and then push it.

        gencfg_service_groups_recluster = self.add_nanny_revision(gencfg_service_groups, self.gencfg_last_rev)

        if not gencfg_service_groups_recluster:
            raise common.errors.TaskFailure("We cannot add revision to specified gencfg_groups. "
                                            "The group with the same revision already exists? "
                                            "Revision:{0}".format(self.gencfg_last_rev))

        # We update groups after self.revision_filename updated
        # so we have all needed changes here for save last snapshot,
        # Activate this snapshot it later.
        snapshots_for_activate = {}
        try:
            logging.info("Step 2.1 - update gencfg instances for nanny groups started. Revision:{0}".format(
                self.gencfg_last_rev))
            for service in self.nanny_groups:
                logging.info("Update gencfg groups for: {0}".format(service))
                # remove old snapshot id
                gencfg_service_groups_recluster[service].pop('snapshot_id', None)

                gencfg_service_groups_recluster[service]['comment'] = "Sandbox auto recluster task. " \
                                                                      "Add revision for gencfg groups to: " \
                                                                      "{recluster_revison}".format(
                    recluster_revison=recluster_rev)

                if not self.dryrun:
                    logging.info("Url: {0}, Action: put, Content: {1}".format(
                        update_nanny_instances_url.format(nanny_url=self.nanny_url,
                                                          servicename=service),
                        gencfg_service_groups_recluster[service]))

                    ret = self.make_auth_req(update_nanny_instances_url.format(nanny_url=self.nanny_url,
                                                                               servicename=service),
                                             token=self.oauth_token,
                                             put=gencfg_service_groups_recluster[service],
                                             timeout=900)
                else:
                    logging.info("Dryrun fake. Url: {0}, Action: put, Content: {1}".format(
                        update_nanny_instances_url.format(nanny_url=self.nanny_url,
                                                          servicename=service),
                        gencfg_service_groups_recluster[service]))
                    # we MUST return snapshot
                    ret = '{"runtime_attrs":{"_id": "dry_run_test_data"}}'

                if not ret:
                    raise common.errors.TaskFailure("Nanny request failed. "
                                                    "Update instances to recluster revision failed")
                logging.info("Step 2.1 - update gencfg instances for nanny group {0} completed. "
                             "Revision:{1}".format(service,
                                                   self.gencfg_last_rev,))
                snapshots_for_activate[service] = json.loads(ret)['runtime_attrs']['_id']

        except Exception as e:
            logging.info("Cannot recluster hosts. Aborting. Error:{0}".format(e))
            raise common.errors.TaskFailure("Cannot recluster hosts. "
                                            "Aborting. "
                                            "Error:{0}".format(e))

        logging.info("Recluster Phase 2 is finished. Hosts going to update now. "
                     "Snapshot for activate: {0}".format(json.dumps(snapshots_for_activate)))

        return snapshots_for_activate


class ReclusterPreparePhase3(ReclusterCtx):

    def __init__(self, oauth_token=False, gencfg_last_rev=False, snapshots_for_activate={}, dryrun=True):
        logging.info("ReclusterPreparePhase3 prepare initialization")
        super(ReclusterPreparePhase3, self).__init__()
        self.oauth_token = oauth_token
        self.snapshots_for_activate = snapshots_for_activate
        self.gencfg_last_rev = gencfg_last_rev
        self.dryrun = dryrun

    def run(self):
        # Phase 3. There we need activate services with LAST snapshot_id.
        # Be carefully - we can commit another changes

        # TODO: Make some test before activation.
        # Check needed searchmaps in mage before activation, check gencfg groups in nanny

        logging.info("Recluster Phase 3 is starting.")

        activate_nanny_url = "{nanny_url}services/{servicename}/events/"
        try:
            for service in self.nanny_groups:
                if not self.dryrun:
                    logging.info("Try to activate service {0} "
                                 "with snapshot_id {1}.".format(service,
                                                                self.snapshots_for_activate[service]))

                    post = {
                        "content": {
                            "snapshot_id": "{0}".format(self.snapshots_for_activate[service]),
                            "state": "ACTIVE",
                            "comment": "Sandbox auto task. "
                                       "Activate {0} snapshot!".format(self.snapshots_for_activate[service]),
                            "recipe": "common"},
                        "type": "SET_SNAPSHOT_STATE"
                    }

                    ret = self.make_auth_req(activate_nanny_url.format(nanny_url=self.nanny_url,
                                                                       servicename=service),
                                             token=self.oauth_token,
                                             post=post,
                                             timeout=900)
                else:
                    logging.info(
                        "Dryrun fake. Try to activate service {0} with snapshot_id {1}.".format(
                            service, self.snapshots_for_activate[service]
                        )
                    )
                    ret = True

                if not ret:
                    raise common.errors.TaskFailure("Nanny request failed. "
                                                    "Cannot SET_SNAPSHOT_STATE to ACTIVE for nanny groups.")

                logging.info("Activating for "
                             "service {0} with snapshot_id {1} completed.".format(service,
                                                                                  self.snapshots_for_activate[service]))

        except Exception as e:
            raise common.errors.TaskFailure("Cannot recluster hosts. Aborting. Error:{0}".format(e))

        logging.info("Recluster Phase 3 is finished. Hosts going to update now.")

        # Set that recluster is finished, if we cannot set it - recluster starts with revision from Mage

        if not self.dryrun:
            mage_set_last_rev_status = self.set_mage_last_revision(self.mage_url,
                                                                   self.gencfg_last_rev,
                                                                   'finished_recluster')
        else:
            logging.info("Dryrun fake. Set mage_set_last_status to:{0}".format('finished_recluster'))
            mage_set_last_rev_status = "dryrun"

        if not mage_set_last_rev_status:
            raise common.errors.TaskFailure("Cannot set mage rev staus. "
                                            "Exiting. Details:{0}".format(mage_set_last_rev_status))

        return True
