import util

import logging
import os
import shutil
import stat

from contextlib import contextmanager

from sandbox.common import errors
from sandbox.sdk2 import ResourceData, Task
from sandbox.sdk2.parameters import LastReleasedResource, String
from sandbox.sdk2.vcs.svn import Arcadia

from sandbox.projects import resource_types as common_resources
from sandbox.projects.common.arcadia.sdk import do_build, mount_arc_path
from sandbox.projects.common.build import parameters as build_params

import sandbox.projects.common.constants as consts


class CommonRequirements(Task.Requirements):
    disk_space = 100 * 1024


class CommonParameters(Task.Parameters):
    arcadia_url = build_params.ArcadiaUrl()

    sym_dump = LastReleasedResource(
        label="dump_syms",
        resource_type=common_resources.BREAKPAD_SYMBOLS_DUMPER,
        required=True
    )

    build_type = String(
        label="Build type",
        required=True,
        group="Build system params",
        default_value=consts.PROFILE_BUILD_TYPE,
        choices=[
            ("Release", consts.RELEASE_BUILD_TYPE),
            ("Debug", consts.DEBUG_BUILD_TYPE),
            ("Profile", consts.PROFILE_BUILD_TYPE),
            ("Coverage", consts.COVERAGE_BUILD_TYPE),
            ("Release with debug info", consts.RELEASE_WITH_DEBUG_INFO_BUILD_TYPE),
            ("Valgrind (debug)", consts.VALGRIND_DEBUG_BUILD_TYPE),
            ("Valgrind (release)", consts.VALGRIND_RELEASE_BUILD_TYPE),
        ]
    )


def gen_symdump(dumper, task, package_dir, target):
    with open(
        os.path.join(
            package_dir,
            "%s.sym" % target[target.rfind("/") + 1:]
        ),
        "w"
    ) as out:
        util.run_process(
            task, "%s %s" % (dumper, target), "sym_dump", stdout=out
        )


@contextmanager
def get_arcadia(arcadia_url):
    try:
        with mount_arc_path(arcadia_url, use_arc_instead_of_aapi=True) as arcadia:
            yield arcadia
    except errors.TaskFailure as e:
        logging.exception(e)
        yield Arcadia.get_arcadia_src_dir(arcadia_url)


def build_package(task, pkg_name, resource_class, base_path, subpaths, sanitize=None):
    build_dir = str(task.path("build_%s" % pkg_name))
    os.mkdir(build_dir)

    arcadia_url = task.Parameters.arcadia_url
    targets = [os.path.join(base_path, subpath) for subpath in subpaths]
    logging.warn(arcadia_url)

    with get_arcadia(arcadia_url) as arcadia_dir:
        do_build(
            build_system="semi_distbuild",
            source_root=arcadia_dir,
            targets=targets,
            build_type=task.Parameters.build_type,
            clear_build=True,
            results_dir=build_dir,
            sanitize=None if sanitize == "none" else sanitize
        )

    parsed_url = Arcadia.parse_url(arcadia_url)
    assert parsed_url.revision
    rev = "r%s" % parsed_url.revision
    versioned_pkg_name = "%s_%s" % (pkg_name, rev)
    logging.warn(versioned_pkg_name)

    package_dir = os.path.join(build_dir, versioned_pkg_name)
    os.mkdir(package_dir)

    dumper_rd = ResourceData(task.Parameters.sym_dump)
    dumper = str(task.path("sym_dump"))
    shutil.copyfile(str(dumper_rd.path), dumper)
    st = os.stat(dumper)
    os.chmod(dumper, st.st_mode | stat.S_IEXEC)

    for subpath in subpaths:
        build_path = os.path.join(build_dir, base_path, subpath)
        target_path = os.path.join(package_dir, os.path.basename(subpath))

        shutil.copyfile(build_path, target_path)

        st = os.stat(target_path)
        os.chmod(target_path, st.st_mode | stat.S_IEXEC)

        if subpath.startswith("daemons"):
            gen_symdump(dumper, task, package_dir, build_path)

    util.create_package(task, os.path.join(build_dir, versioned_pkg_name), resource_class)


def build_daemon_package(task, resource_class, daemon_name, daemon_binary_suffix=None, additional_subpaths=None, sanitize=None):
    subpaths = [
        "daemons/%s/mssngr-%s" % (daemon_name, daemon_name if daemon_binary_suffix is None else daemon_binary_suffix),
        "tools/event_log_dump/event_log_dump",
        "tools/tvm_options_formatter/tvm_options_formatter",
    ]

    if additional_subpaths is None:
        additional_subpaths = [
            "tools/calc_checksum/calc_checksum",
            "tools/compressor/compressor",
            "tools/converter/converter",
            "tools/client/client",
        ]

    build_package(
        task=task,
        pkg_name="mssngr_%s" % daemon_name,
        resource_class=resource_class,
        base_path="mssngr/router",
        subpaths=subpaths + additional_subpaths,
        sanitize=sanitize
    )
