import logging
import os

from sandbox.sdk2 import ResourceData, Task
from sandbox.sdk2.helpers import subprocess as sp, ProcessLog
from sandbox.sdk2.parameters import String

from sandbox.common.errors import TaskFailure


class ArcadiaParameters(Task.Parameters):
    arcadia_url = String(label="arcadia url", required=True)


def run_process(task, cmd, tag, stdout=None, shell=True):
    with ProcessLog(task, logger=logging.getLogger(tag)) as pl:
        if stdout is None:
            stdout = pl.stdout

        code = sp.Popen(cmd, shell=shell, stdout=stdout, stderr=sp.STDOUT).wait()
        if code != 0:
            logging.error(stdout)
            raise Exception("%s exited with code %s" % (cmd, code))


def create_package(task, data_path, resource_class):
    resource_dir = str(task.path("res"))
    try:
        os.mkdir(resource_dir)
    except OSError:
        pass

    data_dir, pkg_name = os.path.split(data_path)
    tar_path = "%s.tar.gz" % os.path.join(resource_dir, pkg_name)
    tar_name = os.path.basename(tar_path)

    if os.path.exists(tar_path):
        raise TaskFailure("Resource %s already exists" % tar_name)

    run_process(task, "cd %s && tar czvf %s %s" % (data_dir, tar_path, pkg_name), "tar")

    resource_class(task, tar_name, tar_path)


def fetch_package(task, resource):
    data = ResourceData(resource)
    data_path = str(data.path)

    run_process(task, "tar xzvf %s" % data_path, "untar")

    return os.path.basename(data_path)[:-7]


def parse_pg_conn_string(value):
    result = {}
    for tok in value.split(" "):
        kv = tok.split("=", 1)
        if len(kv) == 2:
            k, v = kv
            result[k] = v
    return result
