import json
import logging
import requests
from collections import defaultdict

from sandbox import sdk2
from sandbox.projects.common.nanny import nanny
from sandbox.projects.mssngr.rtc import resource_types


class BuildRtcFeedbackReasons(nanny.ReleaseToNannyTask2, sdk2.Task):
    def _get_reasons(self, key):
        url = self.Parameters.tanker_base_url.format(keyset_id=key)
        response = requests.get(url).json()

        _weighted_translations = []

        for key, content in response["keysets"][key]["keys"].items():
            try:
                weight = int(content["info"]["context"])
            except (KeyError, ValueError):
                logging.warning("unknown weight for {}. use default value ({})".format(key, self.Parameters.default_reason_weight))
                weight = self.Parameters.default_reason_weight

            translations = dict()
            for locale, translation in content["translations"].items():
                translations[locale] = translation["form"]

            _weighted_translations.append(
                (weight, key, translations)
            )

        _weighted_translations.sort(key=lambda x: x[0])  # by weight

        result = defaultdict(list)

        for _, key, translations in _weighted_translations:
            for locale, value in translations.items():
                result[locale].append(
                    (key, value)
                )

        return result

    class Parameters(sdk2.Task.Parameters):
        tanker_base_url = sdk2.parameters.String(
            "Tanker URL template",
            default_value="https://tanker-api.yandex-team.ru/keysets/tjson/?keyset-id={keyset_id}&project-id=mssngr_rtc",
            required=True
        )

        default_reason_weight = sdk2.parameters.Integer(
            "Default reason weight",
            default_value=1000,
            required=True
        )

    def on_execute(self):
        data = {
            "AudioReasons": self._get_reasons("call_feedback_audio"),
            "VideoReasons": self._get_reasons("call_feedback_video")
        }
        resource = sdk2.ResourceData(resource_types.FeedbackReasons(
            self, "Feedback reasons from tanker", "feedback_reasons.json"
        ))
        resource.path.write_bytes(json.dumps(data))
