# -*- coding: utf-8 -*-
import json
import logging

from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox.projects.mssngr.rtc import util
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
import sandbox.sandboxsdk.environments as environments


class FeedbackRtcGrepper(sdk2.Task):
    def get_events_by_requests(self, requests, services):
        bad_events = []
        good_events = []
        for r in range(len(requests)):
            request = requests[r]

            logging.info('YQL query: %s', request)

            result = request.run().full_dataframe

            logging.info("Find {} events by request".format(len(result)))

            count_bad_events = 0
            for i, row in result.iterrows():
                event = json.loads(row['value'])
                if (event['score'] <= self.Parameters.bad_mark) and ('text' in event) and (len(event['text'].strip()) > 0):
                    bad_events.append(event)
                    count_bad_events += 1
                else:
                    good_events.append(event)
            self.set_info("Found {} scores total and {} bad scores with description from {}".format(len(result), count_bad_events, services[r]))

        return bad_events, good_events

    def get_bad_events(self, yql_client, date):
        requests = []

        for app_key in [self.Parameters.android_app_key, self.Parameters.ios_app_key]:
            for app in ["superapp", "browser"]:
                request = yql_client.query(
                    query='''
                        SELECT
                            `EventValue` as value
                        FROM hahn.`home/logfeller/logs/{}-metrika-mobile-log/1d/{}`
                        WHERE APIKey = '{}'
                        AND EventType = 'EVENT_CLIENT'
                        AND EventName = 'RTC_USER_CALL_QUALITY_SCORE';
                    '''.format(app, date, app_key), syntax_version=1
                )
                requests.append(request)

        request = yql_client.query(
            query='''
                SELECT `params` as value
                FROM hahn.`logs/bs-watch-log/1d/{}`
                WHERE counterid in ('{}')
                AND url LIKE 'goal://%RTC_USER_CALL_QUALITY_SCORE';
            '''.format(date, str("', '".join([str(i) for i in self.Parameters.web_counter_ids]))), syntax_version=1
        )

        requests.append(request)

        return self.get_events_by_requests(requests, ['ANDROID', 'IOS', 'WEB'])

    def check_child_tasks(self, task_name):
        for call_data in self.Context.calls_data:
            task = sdk2.Task[call_data[task_name]]
            if task.status not in [ctt.Status.SUCCESS, ctt.Status.RELEASED]:
                raise SandboxTaskFailureError("Task {} {} failed".format(task_name, task.id))

    def remove_bad_call_guids(self, bad_call_guids):
        if len(bad_call_guids) == 0:
            return
        self.Context.calls_data = [d for d in self.Context.calls_data if d['event']['call_guid'] not in bad_call_guids]

    def get_log_resources(self, task):
        log_resources = {}
        for r in [
            ['RTC_LOG_MEDIATOR_RESOURCE', 'mediator_log_resource'],
            ['RTC_LOG_WEB_RESOURCE', 'web_log_resource'],
            ['RTC_LOG_ANDROID_RESOURCE', 'andriod_log_resource'],
            ['RTC_LOG_IOS_RESOURCE', 'ios_log_resource'],
            ['RTC_LOG_SIPGW_RESOURCE', 'sipgw_log_resource'],
            ['RTC_LOG_SIPGW_RELAY_RESOURCE', 'sipgw_relay_log_resource'],
        ]:
            resource = sdk2.Resource[r[0]].find(
                task=task,
            ).limit(1)
            if resource.count > 0:
                log_resources[r[1]] = resource.first()

        return log_resources

    def log_info(self, info):
        logging.info(info)
        self.set_info(info)

    class Parameters(sdk2.Task.Parameters):

        date = sdk2.parameters.String("Evaluation date: yyyy-mm-dd; yesterday if empty", default_value='', required=False)
        internal_rtc_queue = sdk2.parameters.String("RTC Queue for messenger internal in startrek or empty", default_value='TEST', required=False)
        external_rtc_queue = sdk2.parameters.String("RTC Queue for messenger external in startrek or empty", default_value='TEST', required=False)
        med_queue = sdk2.parameters.String("Health Queue in startrek or empty", default_value='TEST', required=False)
        bad_mark = sdk2.parameters.Integer("Highest bad score", default_value=3, required=False)

        web_counter_ids = sdk2.parameters.List("Web counter ids", sdk2.parameters.Integer, default=[37738105, 45356799, 48256454], required=False)
        android_app_key = sdk2.parameters.Integer("Andriod application key", default_value=1630031, required=False)
        ios_app_key = sdk2.parameters.Integer("IOS application key", default_value=2126725, required=False)
        create_st_comments = sdk2.parameters.Bool("Call users in the comments", default_value=True, required=False)

        with sdk2.parameters.String("Environment. Debug option") as env:
            env.values.prod = env.Value('Production', default=True)
            env.values.alpha = 'Alpha'
        feedback_limit = sdk2.parameters.Integer("Tasks limit", default_value=0, required=False)
        feedback_offset = sdk2.parameters.Integer("Tasks offset", default_value=0, required=False)

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        environments = [
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yql'),
        ]

    class Caches(sdk2.Requirements.Caches):
        pass

    def on_execute(self):
        with self.memoize_stage['create_log_grepper_tasks']:
            date = util.get_date(self.Parameters.date)
            self.set_info("Collecting feedback for {}".format(date))
            yql_client = util.get_yql_client()

            bad_events, good_events = self.get_bad_events(yql_client, date)
            logging.info("Find {} bad events total".format(len(bad_events)))

            child_tasks = []
            self.Context.calls_data = []
            if self.Parameters.feedback_limit > 0:
                bad_events = bad_events[self.Parameters.feedback_offset:self.Parameters.feedback_offset+self.Parameters.feedback_limit]
            else:
                bad_events = bad_events[self.Parameters.feedback_offset:]

            for event in bad_events:
                if 'text' in event:
                    event['text'] = event['text'].strip()
                else:
                    event['text'] = 'unknown'
                call_guids = [d['event']['call_guid'] for d in self.Context.calls_data]
                if event['call_guid'] in call_guids:
                    ind = call_guids.index(event['call_guid'])
                    self.Context.calls_data[ind]['another_event'] = event
                    continue

                log_grepper_task = sdk2.Task['LOG_RTC_GREPPER'](
                    self,
                    description="{} call guid".format(event['call_guid']),
                    owner=self.owner,
                    date=date,
                    env=self.Parameters.env,
                    call_guid=event['call_guid'],
                    score_meta=json.dumps(event),
                    get_sipgw_logs=False,
                    get_sipgw_relay_logs=False
                )
                log_grepper_task.enqueue()

                child_tasks.append(log_grepper_task)
                self.Context.calls_data.append({'log_grepper_task': log_grepper_task.id, 'event': event, 'bad': False})

            call_guids = [d['event']['call_guid'] for d in self.Context.calls_data]
            for event in good_events:
                if event['call_guid'] in call_guids:
                    ind = call_guids.index(event['call_guid'])
                    self.Context.calls_data[ind]['another_event'] = event

            raise sdk2.WaitTask(child_tasks, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

        with self.memoize_stage['create_log_support_tasks']:
            self.set_info("Parsing mediator logs")
            self.check_child_tasks('log_grepper_task')

            child_tasks = []
            bad_call_guids = []
            for call_data in self.Context.calls_data:
                resource = sdk2.Resource['RTC_LOG_MEDIATOR_RESOURCE'].find(
                    task=sdk2.Task[call_data['log_grepper_task']],
                ).limit(1)
                if resource.count == 0:
                    self.log_info("Remove calls_data for call_guid {} without RTC_LOG_MEDIATOR_RESOURCE".format(call_data['event']['call_guid']))
                    bad_call_guids.append(call_data['event']['call_guid'])
                    continue
                log_resource = resource.first()

                log_support_task = sdk2.Task['RTC_LOG_SUPPORT'](
                    self,
                    description="{} call_guid".format(call_data['event']['call_guid']),
                    owner=self.owner,
                    mediator_log_resource=log_resource,
                    env=self.Parameters.env,
                )
                child_tasks.append(log_support_task)
                log_support_task.enqueue()

                call_data['log_support_task'] = log_support_task.id

            self.remove_bad_call_guids(bad_call_guids)
            raise sdk2.WaitTask(child_tasks, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

        with self.memoize_stage['create_startrek_tasks']:
            self.set_info("Creating startrek tasks")
            self.check_child_tasks('log_support_task')

            child_tasks = []
            bad_call_guids = []
            for call_data in self.Context.calls_data:
                log_resources = self.get_log_resources(sdk2.Task[call_data['log_grepper_task']])

                resources = sdk2.Resource['RTC_CALL_META_INFO_RESOURCE'].find(
                    task=sdk2.Task[call_data['log_support_task']],
                ).limit(1)
                if resources.count == 0:
                    self.log_info("Remove calls_data for call_guid {} without RTC_CALL_META_INFO_RESOURCE".format(call_data['event']['call_guid']))
                    bad_call_guids.append(call_data['event']['call_guid'])
                    continue
                call_meta_resource = resources.first()

                resources = sdk2.Resource['RTC_PLOTS_RESOURCE'].find(
                    task=sdk2.Task[call_data['log_support_task']],
                ).limit(1)
                if resources.count == 0:
                    self.log_info("Remove calls_data for call_guid {} without RTC_PLOTS_RESOURCE".format(call_data['event']['call_guid']))
                    bad_call_guids.append(call_data['event']['call_guid'])
                    continue
                plots_resource = resources.first()

                if 'another_event' in call_data:
                    for k in call_data['another_event']:
                        call_data['event']['another_' + k] = call_data['another_event'][k]

                score_meta_resource = util.RtcScoreMetaInfoResource(self, 'score_meta', 'score_meta_{}_{}.txt'.format(call_data['event']['call_guid'][:10], call_data['event']['user_guid'][:10]))
                resource_data = sdk2.ResourceData(score_meta_resource)
                resource_data.path.write_bytes(json.dumps(call_data['event']))
                resource_data.ready()

                call_meta = json.loads(sdk2.ResourceData(call_meta_resource).path.read_bytes())

                queue = ''
                if call_meta['chat_id'].startswith('0/5/'):
                    queue = self.Parameters.med_queue
                    create_comment = False
                elif call_meta['environment'] == 'mssngr_external':
                    if (len(call_data['event']['text']) == 0) or (call_data['event']['text'] == 'unknown'):
                        self.log_info("Remove calls_data from mssngr_external for call_guid {} without description".format(call_data['event']['call_guid']))
                        bad_call_guids.append(call_data['event']['call_guid'])
                        continue
                    queue = self.Parameters.external_rtc_queue
                    create_comment = False
                elif call_meta['environment'] == 'mssngr_internal':
                    queue = self.Parameters.internal_rtc_queue
                    create_comment = True

                if len(queue) == 0:
                    self.log_info("Remove calls_data for call_guid {} with empty queue".format(call_data['event']['call_guid']))
                    bad_call_guids.append(call_data['event']['call_guid'])
                    continue

                call_guid = call_data['event']['call_guid']

                if not self.Parameters.create_st_comments:
                    create_comment = False

                create_st_task = sdk2.Task['RTC_CREATE_ST_FEEDBACK_TASK'](
                    self,
                    description="{} call_guid".format(call_guid),
                    owner=self.owner,
                    queue=queue,
                    create_comment=create_comment,
                    call_meta_resource=call_meta_resource,
                    plots_resource=plots_resource,
                    score_meta_resource=score_meta_resource,
                    **log_resources
                )
                child_tasks.append(create_st_task)
                create_st_task.enqueue()

                call_data['create_st_task'] = create_st_task.id

            self.remove_bad_call_guids(bad_call_guids)
            raise sdk2.WaitTask(child_tasks, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

        with self.memoize_stage['check_tasks_finished_successfully']:
            self.set_info("Checking that tasks finished successfully")
            self.check_child_tasks('create_st_task')
