# -*- coding: utf-8 -*-
import datetime
import logging

from sandbox import sdk2
import sandbox.sandboxsdk.environments as environments

from sandbox.projects.mssngr.rtc import util


class LogRtcGrepper(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 2 * 60 * 60  # 2 hours

        android_app_key = 1630031
        ios_app_key = 2126725

        web_counter_ids = sdk2.parameters.List(
            "Web counter ids",
            sdk2.parameters.Integer,
            default=[37738105, 45356799, 48256454],
            required=False,
        )
        call_guid = sdk2.parameters.String("Call-guid", required=True)

        date = sdk2.parameters.String(
            "Evaluation date: yyyy-mm-dd. Slow for current date.",
            required=True,
        )

        start_time = sdk2.parameters.String(
            "First YT tabel: hh:mm. Used only for today's date",
            default='00:00',
            required=False,
        )
        end_time = sdk2.parameters.String(
            "Last YT tabel: hh:mm. Used only for today's date",
            default='23:59',
            required=False,
        )

        with sdk2.parameters.String("Environment. Debug option") as env:
            env.values.prod = env.Value('Production', default=True)
            env.values.alpha = 'Alpha'

        get_mediator_logs = sdk2.parameters.Bool("Get mediator logs", default_value=True, required=False)
        get_web_logs = sdk2.parameters.Bool("Get web logs", default_value=True, required=False)
        get_andriod_logs = sdk2.parameters.Bool("Get android logs", default_value=True, required=False)
        get_ios_logs = sdk2.parameters.Bool("Get ios logs", default_value=True, required=False)
        get_sipgw_logs = sdk2.parameters.Bool("Get sipgw logs", default_value=True, required=False)
        get_sipgw_relay_logs = sdk2.parameters.Bool("Get sipgw-relay logs", default_value=True, required=False)

    def get_from_(self, table):
        if self.Parameters.date != datetime.date.today().strftime("%Y-%m-%d"):
            from_ = """hahn.`{}/1d/{}`""".format(table, self.Parameters.date)

        else:
            from_ = """RANGE("{}/stream/5min", "{}T{}:00", "{}T{}:00")""".format(
                table, self.Parameters.date, self.Parameters.start_time, self.Parameters.date, self.Parameters.end_time
            )

        logging.info("From = {}".format(from_))
        return from_

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        environments = [
            environments.PipEnvironment("yandex-yt"),
            environments.PipEnvironment("yql"),
        ]

    class Caches(sdk2.Requirements.Caches):
        pass

    def on_execute(self):
        call_guid = self.Parameters.call_guid
        yql_client = util.get_yql_client()

        if self.Parameters.get_mediator_logs:
            from_ = self.get_from_("home/logfeller/logs/rtc-{}-mediator-log".format(self.Parameters.env))
            mediator_logs = util.get_backend_strings_logs(yql_client, call_guid, from_, 'mediator')
            if len(mediator_logs) > 0:
                util.create_resource(self, 'mediator_logs', mediator_logs, util.RtcLogMediatorResource)

        if self.Parameters.get_web_logs:
            from_ = self.get_from_("logs/bs-watch-log")
            web_logs = util.get_web_logs(yql_client, call_guid, from_, self.Parameters.web_counter_ids)
            if len(web_logs) > 0:
                util.create_resource(self, 'web_logs', web_logs, util.RtcLogWebResource)

        for app in ["superapp", "browser"]:
            if self.Parameters.get_andriod_logs:
                from_ = self.get_from_("home/logfeller/logs/{}-metrika-mobile-log".format(app))
                android_logs = util.get_logs_from_appmetrica(yql_client, call_guid, from_, self.Parameters.android_app_key, 'android_')
                if len(android_logs) > 0:
                    util.create_resource(self, '{}_android_logs'.format(app), android_logs, util.RtcLogAndroidResource)

            if self.Parameters.get_ios_logs:
                from_ = self.get_from_("home/logfeller/logs/{}-metrika-mobile-log".format(app))
                ios_logs = util.get_logs_from_appmetrica(yql_client, call_guid, from_, self.Parameters.ios_app_key, 'IOS_')
                if len(ios_logs) > 0:
                    util.create_resource(self, '{}_ios_logs'.format(app), ios_logs, util.RtcLogIosResource)

        if self.Parameters.get_sipgw_logs:
            from_ = self.get_from_("home/logfeller/logs/rtc-{}-sipgw-log".format(self.Parameters.env))
            sipgw_logs = util.get_backend_strings_logs(yql_client, call_guid, from_, 'sipgw')
            if len(sipgw_logs) > 0:
                util.create_resource(self, 'sipgw_logs', sipgw_logs, util.RtcLogSipgwResource)

        if self.Parameters.get_sipgw_relay_logs:
            from_ = self.get_from_("home/logfeller/logs/rtc-{}-sipgw-relay-log".format(self.Parameters.env))
            sipgw_relay_logs = util.get_backend_strings_logs(yql_client, call_guid, from_, 'sipgw-relay')
            if len(sipgw_relay_logs) > 0:
                util.create_resource(self, 'sipgw_relay_logs', sipgw_relay_logs, util.RtcLogSipgwRelayResource)
