# -*- coding: utf-8 -*-
from collections import defaultdict
import json
import logging
import os
import requests
import zipfile

from sandbox import sdk2
from sandbox.projects.common import file_utils as fu
import sandbox.sandboxsdk.environments as environments

from sandbox.projects.mssngr.rtc import util


class RtcCreateStFeedbackTask(sdk2.Task):
    def add_resource_attachment_to_startrek(self, resource, file_name, startrek_auth):
        data = sdk2.ResourceData(resource).path.read_bytes()
        url = "https://st-api.yandex-team.ru/v2/attachments?filename={}.zip".format(file_name)
        files = {'file': data}
        response = requests.post(url, headers=startrek_auth, files=files)

        logging.info('Adding attach to startrek. Response: {}'.format(response.text))

        attach_id = json.loads(response.text)['id']
        self.set_info("Added attach {} to startrek, id {}".format(file_name, attach_id))
        return attach_id

    def write_login(self, login):
        if self.environment == 'mssngr_internal':
            return '@' + login
        return login

    def set_startrek_info(self, call, event):
        call['followers'] = ['likaleonovich']
        for pl, resp in self.Parameters.responsible.items():
            if pl in call['platforms']:
                call['followers'].append(resp)

        if event['user_guid'] == call['caller_guid']:
            event['user_login'] = call['caller_login']
            call['assignee'] = self.Parameters.responsible[call['caller_platform']]
        elif event['user_guid'] == call['callee_guid']:
            event['user_login'] = call['callee_login']
            call['assignee'] = self.Parameters.responsible[call['callee_platform']]
        else:
            event['user_login'] = 'unknown'
            call['assignee'] = 'robot-rtc-logs-grepr'

        call['summary'] = "[{}] {}: {} => {}: {}".format(event['datetime'], event['user_login'], call['caller_platform'], call['callee_platform'], event['text'][0: 40])
        call['description'].append("Call_guid: {}".format(event['call_guid']))
        call['description'].append("User {} rated **{}**, user_guid: {}".format(self.write_login(event['user_login']), event['score'], event['user_guid']))
        if self.Parameters.create_comment:
            call['followers'].append(event['user_login'])

        call['description'].append("")
        call['description'].append("<[{}]>".format(event['text']))
        if (event['user_reasons'] != 'unknown') and (event['user_reasons'] != []):
            call['description'].append("reasons: {}".format(json.dumps(event['user_reasons'])))
        call['description'].append("")

        if event['another_score'] != 'unknown':
            if event['another_user_guid'] == call['caller_guid']:
                event['another_user_login'] = call['caller_login']
            elif event['another_user_guid'] == call['callee_guid']:
                event['another_user_login'] = call['callee_login']
            call['description'].append("User {} rated **{}**, user_guid: {}".format(self.write_login(event['another_user_login']), event['another_score'], event['another_user_guid']))
            if self.Parameters.create_comment:
                call['followers'].append(event['another_user_login'])

            call['description'].append("")
            call['description'].append("<[{}]>".format(event['another_text']))
            if (event['another_user_reasons'] != 'unknown') and (event['another_user_reasons'] != []):
                call['description'].append("reasons: {}".format(json.dumps(event['another_user_reasons'])))
            call['description'].append("")

        call['description'].append("Caller login: {}, caller guid: {}".format(self.write_login(call['caller_login']), call['caller_guid']))
        if call['is_caller_relay']:
            call['description'].append("Relay, ip: {}".format(call['caller_ip']))
        if call['caller_vpn_enabled']:
            call['description'].append("Caller's VPN is enabled")
        call['description'].append("<{Caller device:\n%%")
        call['description'].append(json.dumps(call['caller_device_info'], indent=4))
        call['description'].append("%%\n}>")
        call['description'].append("")
        call['description'].append("")
        call['description'].append("Callee login: {}, callee guid: {}".format(self.write_login(call['callee_login']), call['callee_guid']))
        if call['is_callee_relay']:
            call['description'].append("Relay, ip: {}".format(call['callee_ip']))
        if call['callee_vpn_enabled']:
            call['description'].append("Callee's VPN is enabled")
        call['description'].append("<{Callee device:\n%%")
        call['description'].append(json.dumps(call['callee_device_info'], indent=4))
        call['description'].append("%%\n}>")
        call['description'].append("")
        call['description'].append("Call created at {}".format(call['call_created']))
        call['description'].append("Call duration {}".format(call['call_duration']))

        if self.Parameters.plots_resource:
            zf = zipfile.ZipFile(str(sdk2.ResourceData(self.Parameters.plots_resource).path), 'r')
            zip_files = zf.namelist()

            for name_prefix in ['caller_', 'callee_']:
                call['description'].append("")
                for stats_prefix in ['inbound_audio_stream', 'inbound_audio_track', 'inbound_video_stream', 'inbound_video_track', 'outbound_audio_stream',
                                      'outbound_audio_track', 'outbound_video_stream', 'outbound_video_track', 'ice_candidate_pair']:
                    prefix = name_prefix + stats_prefix
                    call['description'].append("<{" + prefix + ":")
                    for plot_name in zip_files:
                        if plot_name.startswith(prefix):
                            call['description'].append("  <{" + plot_name + ":")
                            call['description'].append("https://proxy.sandbox.yandex-team.ru/{}/{}".format(self.Parameters.plots_resource.id, plot_name))
                            call['description'].append("}>")

                    call['description'].append("}>")

    def add_resources_to_startrek(self, startrek_auth):
        attaches = [
            self.add_resource_attachment_to_startrek(
                self.Parameters.mediator_log_resource,
                'mediator_log',
                startrek_auth
            )
        ]

        if self.Parameters.web_log_resource:
            attaches.append(
                self.add_resource_attachment_to_startrek(
                    self.Parameters.web_log_resource,
                    'web_log',
                    startrek_auth
                )
            )

        if self.Parameters.andriod_log_resource:
            attaches.append(
                self.add_resource_attachment_to_startrek(
                    self.Parameters.andriod_log_resource,
                    'android_log',
                    startrek_auth
                )
            )

        if self.Parameters.ios_log_resource:
            attaches.append(
                self.add_resource_attachment_to_startrek(
                    self.Parameters.ios_log_resource,
                    'ios_log',
                    startrek_auth
                )
            )

        if self.Parameters.sipgw_log_resource:
            attaches.append(
                self.add_resource_attachment_to_startrek(
                    self.Parameters.sipgw_log_resource,
                    'sipgw_log',
                    startrek_auth
                )
            )

        if self.Parameters.sipgw_relay_log_resource:
            attaches.append(
                self.add_resource_attachment_to_startrek(
                    self.Parameters.sipgw_relay_log_resource,
                    'sipgw_relay_log',
                    startrek_auth
                )
            )

        return attaches

    def create_startrek_task(self, call, event, startrek_auth):
        url = "https://st-api.yandex-team.ru/v2/issues"

        if (self.Parameters.queue == 'TEST'):
            call['assignee'] = 'robot-rtc-logs-grepr'
            call['followers'] = []
            event['user_login'] = 'robot-rtc-logs-grepr'

        tags = ['call_feedback', 'call_guid_{}'.format(event['call_guid'])]
        if event['user_reasons'] != 'unknown':
            tags += [tag.lower() for tag in event['user_reasons']]

        ticket_data = {'queue': self.Parameters.queue,
                       'summary': call['summary'],
                       'assignee': call['assignee'],
                       'description': str('\n'.join(call['description'])),
                       'tags': tags,
                       'followers': list(set(call['followers'])),
                       'attachmentIds': call['attachment_ids']
                       }

        logging.debug('Trying to create tiket.\nCall: {}\nTiket: {}'.format(call, ticket_data))

        response = requests.post(url, headers=startrek_auth, json=ticket_data)
        logging.info('Creating startrek task. Response: {}'.format(response.text))
        task_key = json.loads(response.text)['key']
        if response.ok:
            self.set_info("Create task https://st.yandex-team.ru/{}/ for call_guid {}".format(task_key, event['call_guid']))
        return task_key

    def create_need_info_comment(self, call, event, task_key, startrek_auth):
        if event['user_login'] in fu.read_lines(os.path.join(os.path.dirname(__file__), 'muted_users.txt')):
            self.set_info("Don't create comment for muted user {}".format(event['user_login']))
            return

        text = (
            "Спасибо за фидбэк о звонке.\n"
            "" + "{} | {} <-> {}".format(call['iso_eventtime'], call['caller_login'], call['callee_login']) + "\n"
            "Можешь рассказать подробнее про возникшую неполадку:\n\n"
            "1. Насколько плохо все было, устранилась ли неполадка после перезвона?\n"
            "2. Наблюдались ли похожие проблемы у собеседников?\n"
            "3. Оценка оставлена именно за тот сеанс, на котором произошли неполадки?\n"
            "4. Условия звонка: интернет-соединение и др.\n\n"
            "Любая дополнительная информация поможет нам лучше понять причины.\n\n"
            "Если ты не хочешь больше получать призывы в очередь RTCSUP о фидбэке звонка,"
            "добавь свой логин новой строчкой в файл"
            " https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/mssngr/rtc/RtcCreateStFeedbackTask/muted_users.txt"
        )
        commentary = {"text": text, "summonees": [event['user_login']]}
        create_comment_url = "https://st-api.yandex-team.ru/v2/issues/{}/comments".format(task_key)
        response = requests.post(create_comment_url, headers=startrek_auth, json=commentary)
        logging.debug("Create commentary {}. Response: {}".format(task_key, response.text))

        update_status_url = 'https://st-api.yandex-team.ru/v2/issues/{}/transitions/need_info/_execute'.format(task_key)
        response = requests.post(update_status_url, headers=startrek_auth)
        logging.debug('Update status need_info to task {}. Response: {}'.format(task_key, response.text))
        self.set_info("Comment was created")

    class Parameters(sdk2.Task.Parameters):
        score_meta_resource = sdk2.parameters.Resource(
            'Call score meta information',
            resource_type=util.RtcScoreMetaInfoResource,
            required=True
        )

        call_meta_resource = sdk2.parameters.Resource(
            'Call meta information',
            resource_type=util.RtcCallMetaInfoResource,
            required=True
        )

        mediator_log_resource = sdk2.parameters.Resource(
            'Mediator logs zip',
            resource_type=util.RtcLogMediatorResource,
            required=True
        )

        web_log_resource = sdk2.parameters.Resource(
            'Web logs zip',
            resource_type=util.RtcLogWebResource,
            required=False
        )

        andriod_log_resource = sdk2.parameters.Resource(
            'Android logs zip',
            resource_type=util.RtcLogAndroidResource,
            required=False
        )

        ios_log_resource = sdk2.parameters.Resource(
            'IOS logs zip',
            resource_type=util.RtcLogIosResource,
            required=False
        )

        sipgw_log_resource = sdk2.parameters.Resource(
            'SipGW logs zip',
            resource_type=util.RtcLogSipgwResource,
            required=False
        )

        sipgw_relay_log_resource = sdk2.parameters.Resource(
            'SipGW relay logs zip',
            resource_type=util.RtcLogSipgwRelayResource,
            required=False
        )

        plots_resource = sdk2.parameters.Resource(
            'Plots zip',
            resource_type=util.RtcPlotsResource,
            required=False
        )

        responsible = {
            'IOS': 'a-nezhelskoy',
            'WEB': 'karserg',
            'ANDROID': 'weird',
        }

        queue = sdk2.parameters.String("Queue in startrek", default_value='TEST', required=True)
        create_comment = sdk2.parameters.Bool("Call the user in the comments", default_value=False, required=False)

        plot_names = [
            'caller_audio_packets_lost.png',
            'callee_audio_packets_lost.png',
            'caller_audio_concealed_samples.png',
            'callee_audio_concealed_samples.png',
            'caller_round_trip_time.png',
            'callee_round_trip_time.png',
            'caller_video_frames_dropped.png',
            'callee_video_frames_dropped.png',
            'caller_video_packets_lost.png',
            'callee_video_packets_lost.png',
        ]

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        environments = [
            environments.PipEnvironment("yandex-yt"),
            environments.PipEnvironment("yql"),
        ]

    class Caches(sdk2.Requirements.Caches):
        pass

    def on_execute(self):
        call = json.loads(sdk2.ResourceData(self.Parameters.call_meta_resource).path.read_bytes())
        score_meta = json.loads(sdk2.ResourceData(self.Parameters.score_meta_resource).path.read_bytes())
        event = defaultdict(lambda: "unknown", score_meta)
        self.environment = call['environment']

        startrek_auth = util.get_startrek_auth()

        call['attachment_ids'] = self.add_resources_to_startrek(startrek_auth)

        self.set_startrek_info(call, event)

        self.create_startrek_task(call, event, startrek_auth)
