import datetime
import json
import logging
import string
import StringIO
import random
import requests
import time
import zipfile
from collections import defaultdict
from sandbox import sdk2


def get_yql_client():
    from yql.api.v1.client import YqlClient

    token = sdk2.Vault.data("robot-rtc-logs-grepr-yql-token")
    return YqlClient(token=token, db="hahn")


def get_messenger_auth():
    token = sdk2.Vault.data("robot-rtc-logs-grepr-messenger-token")
    return {"Authorization": "OAuthTeam {}".format(token)}


def get_startrek_auth():
    token = sdk2.Vault.data("robot-rtc-logs-grepr-startrek-token")
    return {"Authorization": "OAuth {}".format(token)}


def get_staff_nicknames(guids, messenger_auth):
    url = "https://api.messenger.yandex-team.ru/api/"

    data = {"method": "get_users_data", "params": {"guids": guids}}
    response = requests.post(url, headers=messenger_auth, json=data)
    logging.info("Get staff logins. Response: {}".format(response.__dict__))

    logins = defaultdict(lambda: "unknown")
    if response.text == '':
        return logins
    for user in json.loads(response.text)["data"]["users"]:
        logins[user["guid"]] = user["nickname"]

    return logins


class MetaApiHelper:
    def __init__(self, env):
        self.env = env
        if env == 'prod':
            self.url = 'https://messenger-internal.yandex.net/meta_api/'
        else:
            self.url = 'https://messenger-internal.alpha.yandex.net/meta_api/'

        self.feedback_grepper_tvm_id = 2017879
        if env == 'prod':
            self.messenger_tvm_id = 2000482  # mssng meta api prod
        else:
            self.messenger_tvm_id = 2001664  # mssng meta api alpha

    def get_yandex_nickname(self, user_guid, tvm_ticket):
        response = requests.post(
            url=self.url,
            headers={
                'X-Ya-Service-Ticket': tvm_ticket,
                'X-REQUEST-ID': ''.join(random.choice(string.ascii_letters) for _ in range(20)),
            },
            json={
                'method': 'get_user_info',
                'params': {'guid': user_guid}
            },
        )

        logging.info('Get user info for user_guid {}; response status_code: {}; reason: {}'.format(user_guid, response.__dict__['status_code'], response.__dict__['reason']))

        try:
            return response.json()['data']['login']
        except KeyError:
            logging.warn('can not find login for user_guid {}'.format(user_guid))
            return 'unknown'

    def get_yandex_nicknames(self, user_guids):
        tvm_ticket = self.get_tvm_ticket()
        nicknames = defaultdict(lambda: "unknown")
        for guid in user_guids:
            nicknames[guid] = self.get_yandex_nickname(guid, tvm_ticket)
        logging.info("Get yandex nicknames from {} for {}".format(self.env, user_guids))
        return nicknames

    def get_tvm_ticket(self):
        import ticket_parser2
        import ticket_parser2.api.v1

        secret = sdk2.yav.Secret('sec-01dyq7j7kf4m0tfczdkev68bad').data()['client_secret']  # feedback_grepper tvm secret
        ts = int(time.time())

        tvm_keys = requests.get(
            'https://tvm-api.yandex.net/2/keys?lib_version=%s' % (ticket_parser2.__version__, )
        ).content

        svc_ctx = ticket_parser2.api.v1.ServiceContext(self.feedback_grepper_tvm_id, secret, tvm_keys)

        ticket_response = requests.post(
            'https://tvm-api.yandex.net/2/ticket/',
            data={
                'grant_type': 'client_credentials',
                'src': self.feedback_grepper_tvm_id,
                'dst': self.messenger_tvm_id,
                'ts': ts,
                'sign': svc_ctx.sign(ts, self.messenger_tvm_id)
            }
        )

        logging.info(ticket_response.__dict__)
        return ticket_response.json()[str(self.messenger_tvm_id)]['ticket']


def get_backend_logs(yql_client, call_guid, from_):
    request = yql_client.query(
        query="""
            SELECT *
            FROM {}
            WHERE call_guid == "{}"
            ORDER BY ts;
        """.format(
            from_, call_guid
        ),
        syntax_version=1,
    )

    logging.info("YQL query: %s", request)

    return request.run().full_dataframe


def convert_back_log_to_string(logs):
    logs_jsons = [row.dropna().to_json() for i, row in logs.iterrows()]
    return str("\n".join(logs_jsons))


def get_backend_strings_logs(yql_client, call_guid, from_, service):
    logs = get_backend_logs(yql_client, call_guid, from_)
    if len(logs) == 0:
        return []

    return [{'name': service + '_logs.txt', 'data': convert_back_log_to_string(logs)}]


def get_logs_from_appmetrica(
    yql_client, call_guid, from_, application_id, name_prefix
):
    request = yql_client.query(
        query="""
            SELECT
                EventValue as value,
                EventName as event
            FROM {}
            WHERE APIKey = '{}'
            AND EventType = 'EVENT_CLIENT'
            AND EventName LIKE 'RTC_%'
            AND Yson::Contains(Yson::ParseJson(EventValue), "call_guid")
            AND Yson::LookupString(Yson::ParseJson(EventValue), "call_guid") == "{}";
        """.format(
            from_, application_id, call_guid
        ),
        syntax_version=1,
    )

    return get_logs_by_request(request, name_prefix)


def logs_comparator(v):
    if "datetime" in v:
        return v["datetime"]
    return ""


def get_user_guid(l):
    if 'user_guid' in l:
        return l['user_guid'][:4]
    return ''


def get_session_id(l):
    if 'session_id' in l:
        return l['session_id']
    return ''


def get_session_num(l, user_sessions):
    session_id = get_session_id(l)
    if session_id in user_sessions:
        ind = user_sessions.index(session_id)
    else:
        user_sessions.append(session_id)
        ind = len(user_sessions) - 1
    if ind == 0:
        return ''
    return '_' + str(ind)


def get_logs_filename(l, name_prefix, user_sessions):
    return name_prefix + get_user_guid(l) + get_session_num(l, user_sessions) + '.txt'


def get_logs_by_request(request, name_prefix):
    logging.info("YQL query: %s", request)

    result = request.run().full_dataframe

    logging.info("Find {} logs rows".format(len(result)))

    logs = []
    for i, row in result.iterrows():
        d = json.loads(row["value"])
        d["event_name"] = row["event"]
        logs.append(d)

    sorted_logs = sorted(logs, key=logs_comparator)

    out = defaultdict(list)
    user_sessions = defaultdict(list)
    for l in sorted_logs:
        out[get_logs_filename(l, name_prefix, user_sessions[get_session_id(l)])].append(json.dumps(l))

    return [{'name': n, 'data': str("\n".join(out[n]))} for n in out]


def get_web_logs(yql_client, call_guid, from_, web_counter_ids=[]):
    request = yql_client.query(
        query="""
            SELECT
                params as value,
                Re2::Capture(@@[^\/]+$@@)(`url`)._0 as event
            FROM {}
            WHERE counterid in ('{}')
            AND url LIKE 'goal://%RTC_%'
            AND Yson::Contains(Yson::ParseJson(params), "call_guid")
            AND Yson::LookupString(Yson::ParseJson(params), "call_guid") == "{}";
        """.format(
            from_, str("', '".join([str(i) for i in web_counter_ids])), call_guid
        ),
        syntax_version=1,
    )

    return get_logs_by_request(request, 'web_')


def get_date(date):
    return (
        date
        if date != ""
        else (datetime.date.today() - datetime.timedelta(days=1)).strftime("%Y-%m-%d")
    )


def create_resource(task, file_name, file_datas, resource_type):
    f = StringIO.StringIO()
    zf = zipfile.ZipFile(f, 'w')
    for i in range(len(file_datas)):
        if file_datas[i] == {}:
            continue
        zf.writestr(file_datas[i]['name'], file_datas[i]['data'])
    zf.close()
    f.read()

    resource = resource_type(task, file_name, "{}.zip".format(file_name))
    resource_data = sdk2.ResourceData(resource)
    resource_data.path.write_bytes(f.buf)
    f.close()
    resource_data.ready()
    logging.debug("Created '{}.zip' logs file".format(file_name))
    return resource.id


class RtcLogMediatorResource(sdk2.Resource):
    """ Rtc log mediator resource """


class RtcLogWebResource(sdk2.Resource):
    """ Rtc log web resource """


class RtcLogAndroidResource(sdk2.Resource):
    """ Rtc log mediator resource """


class RtcLogIosResource(sdk2.Resource):
    """ Rtc log ios resource """


class RtcLogSipgwResource(sdk2.Resource):
    """ Rtc log SipGW resource """


class RtcLogSipgwRelayResource(sdk2.Resource):
    """ Rtc log SipGW relay resource """


class RtcCallMetaInfoResource(sdk2.Resource):
    """ Rtc call meta info resource """


class RtcScoreMetaInfoResource(sdk2.Resource):
    """ Rtc score meta info resource """


class RtcPlotsResource(sdk2.Resource):
    """ Rtc plots resource """
    ttl = 'inf'
