import os.path
import shutil
import logging

from sandbox.projects.mssngr.common import util
from sandbox.projects.mssngr.runtime import resources

from sandbox import sdk2
from sandbox.sdk2.parameters import LastReleasedResource, List, Resource, String

from sandbox.projects.common.nanny import nanny


class BuildMssngrRouterPartitionMap(nanny.ReleaseToNannyTask2, sdk2.Task):
    """Builds partition map for router balancers and workers"""

    cluster_to_groups = {
        "testing": ["gencfg:MAN_TESTING_MSSNGR_ROUTER_WORKERS", "gencfg:SAS_TESTING_MSSNGR_ROUTER_WORKERS", "gencfg:VLA_TESTING_MSSNGR_ROUTER_WORKERS"],
        "alpha": ["gencfg:MAN_ALPHA_MSSNGR_ROUTER_WORKERS", "gencfg:SAS_ALPHA_MSSNGR_ROUTER_WORKERS", "gencfg:VLA_ALPHA_MSSNGR_ROUTER_WORKERS"],
        "production": ["gencfg:MAN_MSSNGR_ROUTER_WORKERS", "gencfg:SAS_MSSNGR_ROUTER_WORKERS", "gencfg:VLA_MSSNGR_ROUTER_WORKERS"]
    }

    cluster_to_resources = {
        "testing": (resources.MssngrTestingRouterPrevPartitionMap, resources.MssngrTestingRouterPartitionMap),
        "alpha": (resources.MssngrAlphaRouterPrevPartitionMap, resources.MssngrAlphaRouterPartitionMap),
        "production": (resources.MssngrProductionRouterPrevPartitionMap, resources.MssngrProductionRouterPartitionMap),
        "none": (resources.MssngrRouterPrevPartitionMap, resources.MssngrRouterPartitionMap),
    }

    all_resources = [
        r
        for k, v in cluster_to_resources.items()
        for r in v

    ]

    class Requirements(sdk2.Task.Requirements):
        pass

    class Parameters(sdk2.Task.Parameters):
        gencfg_groups = List(
            label="enter gencfg groups",
        )

        with sdk2.parameters.RadioGroup("OR select cluster") as cluster:
            cluster.values.production = cluster.Value("production")
            cluster.values.alpha = cluster.Value("alpha")
            cluster.values.testing = cluster.Value("testing")
            cluster.values.none = cluster.Value("not selected", default=True)

        gencfg_branch = String(
            label="gencfg branch (default - trunk)",
        )

        replace_map = List(
            label="List of host replacements host:geo:host1=host:geo:host2"
        )

        replica_count = String(
            label="replica count for shards",
            default="3"
        )

        config_generator = LastReleasedResource(
            label="config generator",
            resource_type=resources.MssngrRouterConfigGenerator,
            required=True
        )

        prev_partition_map = Resource(
            label="prev partition map",
            resource_type=[
                resources.MssngrAlphaRouterPrevPartitionMap,
                resources.MssngrAlphaRouterPartitionMap,
                resources.MssngrRouterPrevPartitionMap,
                resources.MssngrRouterPartitionMap,
                resources.MssngrTestingRouterPrevPartitionMap,
                resources.MssngrTestingRouterPartitionMap,
                resources.MssngrProductionRouterPrevPartitionMap,
                resources.MssngrProductionRouterPartitionMap
            ],
            required=False
        )

    def on_execute(self):
        package_path = util.fetch_package(self, self.Parameters.config_generator)
        config_generator_path = os.path.join(package_path, "router_config_generator")

        if self.Parameters.prev_partition_map:
            prev_path = "%s" % sdk2.ResourceData(self.Parameters.prev_partition_map).path
            prev_partition_map_resource = self.cluster_to_resources[self.Parameters.cluster][0]
            prev_map = prev_partition_map_resource(self, "prev partition map", "prev_partition_map.json")
            shutil.copyfile(prev_path, "%s" % sdk2.ResourceData(prev_map).path)
        else:
            prev_path = None

        new_partition_map_resource = self.cluster_to_resources[self.Parameters.cluster][1]

        new_partition_map = new_partition_map_resource(self, "partition map", "partition_map.json")
        new_partition_map_data = sdk2.ResourceData(new_partition_map)

        gencfg_groups = self.Parameters.gencfg_groups

        logging.info("gencfg groups are: {}".format(gencfg_groups))

        if self.Parameters.cluster != "none" and not gencfg_groups:
            gencfg_groups = self.cluster_to_groups[self.Parameters.cluster]

        logging.info("gencfg groups are: {}".format(gencfg_groups))

        groups = " ".join("-g %s" % g for g in gencfg_groups)

        replacements = ""
        if self.Parameters.replace_map:
            replacements = " ".join("-m {}".format(r) for r in self.Parameters.replace_map)
        branch = "--branch %s" % self.Parameters.gencfg_branch if self.Parameters.gencfg_branch else ""
        input_path = ("-i " + prev_path) if prev_path else ""

        util.run_process(self, "{} partition_map {} {} {} -o {} -r {} {}".format(
            config_generator_path,
            groups,
            branch,
            input_path,
            new_partition_map_data.path,
            self.Parameters.replica_count,
            replacements
        ), "gen")
