# -*- coding: utf-8 -*-

import datetime
import json
import logging
import os

from collections import namedtuple

from sandbox import sdk2
from sandbox.common.errors import VaultNotFound
from sandbox.common.types.resource import State
from sandbox.common.types.task import ReleaseStatus
from sandbox.sandboxsdk import process

from sandbox.projects.common import utils
from sandbox.projects.mssngr.common.util import fetch_package, parse_pg_conn_string
from sandbox.projects.mssngr.runtime import resources


Env = namedtuple("Env", "config_type release_status secret_name")

ENVS = {
    "testing": Env(resources.MssngrRouterWorkerTestingConfig, ReleaseStatus.TESTING, "mssngr_db_mssngrdb_test"),
    "alpha": Env(resources.MssngrRouterWorkerAlphaConfig, ReleaseStatus.PRESTABLE, "mssngr_db_mssngrdb_alpha"),
    "production": Env(resources.MssngrRouterWorkerConfig, ReleaseStatus.STABLE, "mssngr_db_mssngrdb_prod_0"),
}


class BaseParameters(sdk2.Parameters):
    with sdk2.parameters.RadioGroup("Environment") as env:
        env.values["testing"] = env.Value(value="testing", default=True)
        env.values["alpha"] = env.Value(value="alpha")
        env.values["production"] = env.Value(value="production")

    resource_ttl = sdk2.parameters.Integer(
        "Resource TTL (days)",
        default=1,
    )

    yt_proxy = sdk2.parameters.String(
        "YT proxy",
        default="locke",
    )

    yt_path = sdk2.parameters.String(
        "YT path for bstr",
        default="//home/mssngr/state_cache_bstr",
    )

    custom_mode = sdk2.parameters.Integer(
        "Bitmask containing set of entities to fetch",
        default=None,
    )

    banned_pg_location = sdk2.parameters.String(
        "Disable Postgres location",
        default=None,
    )

    bstr = sdk2.parameters.LastReleasedResource(
        "Bstr binary",
        resource_type=resources.MssngrBstr,
        state=(State.READY),
        required=True,
        attrs=dict(released='stable'),
    )

    pusher = sdk2.parameters.LastReleasedResource(
        "Cache pusher binary",
        resource_type=resources.MssngrRouterStateCachePusher,
        state=(State.READY),
        required=True,
        attrs=dict(released='stable'),
    )


class MssngrDeployStateCache(sdk2.Task):
    """Builds and deploys mssngr-worker state cache"""

    class Requirements(sdk2.Task.Requirements):
        disk_space = 50 * 1024  # 50 GB
        ram = 48 * 1024         # 48 Gb

    class Parameters(BaseParameters):
        cacher = sdk2.parameters.LastReleasedResource(
            "State cacher binary",
            resource_type=resources.MssngrRouterStateCacher,
            state=(State.READY),
            required=True,
        )

    def _get_from_vault(self, key):
        try:
            value = sdk2.Vault.data(self.owner, key)
        except VaultNotFound:
            value = sdk2.Vault.data("MSSNGR", key)
        return str(value)

    def _process_config(self, worker_config_path, conn_params, banned_loc):
        data = json.load(open(worker_config_path, "r"))
        result = os.path.join(os.path.dirname(worker_config_path), "postgres_config.json")
        pg_config = data["postgres_config"]

        for p in ["user"]:
            pg_config[p] = conn_params[p]

        if banned_loc:
            banned_loc = banned_loc.strip().lower()
            pg_config["servers"] = [s for s in pg_config["servers"] if s.get("location") != banned_loc]
            logging.warn("Ban Postgres location: {}".format(banned_loc))

        config = json.dumps(pg_config)
        logging.debug("Using postgres config: {}".format(config))
        with open(result, "w") as f:
            f.write(config)
        return result

    def on_execute(self):
        env_name = self.Parameters.env
        env = ENVS[env_name]

        config_id = utils.get_and_check_last_released_resource_id(env.config_type, env.release_status)

        work_dir = str(self.path("data"))
        os.mkdir(work_dir)

        bstr_bin_path = str(sdk2.ResourceData(self.Parameters.bstr).path)
        cacher_bin_path = os.path.join(fetch_package(self, self.Parameters.cacher), "mssngr-state-cacher")

        pusher_bin_path = os.path.join(fetch_package(self, self.Parameters.pusher), "mssngr-state-cache-pusher")

        conn_params = parse_pg_conn_string(self._get_from_vault(env.secret_name))
        worker_config_path = self._process_config(
            os.path.join(fetch_package(self, sdk2.Resource[config_id]), "templates", "mssngr-router.json"),
            conn_params,
            self.Parameters.banned_pg_location
        )

        run_args = [
            pusher_bin_path,
            "--bstr", bstr_bin_path,
            "--cacher", cacher_bin_path,
            "--config", worker_config_path,
            "--dir", work_dir,
            "--yt-proxy", self.Parameters.yt_proxy,
            "--yt-path", "{}/{}".format(self.Parameters.yt_path, env_name),
            "--cleanup",
            "--verbose",
        ]

        if self.Parameters.custom_mode:
            run_args += ["--mode", str(self.Parameters.custom_mode)]

        process.run_process(
            run_args,
            environment={
                "YT_TOKEN": self._get_from_vault("yt-token-{}".format(self.Parameters.yt_proxy)),
                "PGAAS_PASSWORD": conn_params["password"],
            },
            log_prefix="state_cache_pusher"
        )

        descr = "{} state cache {}".format(env_name, datetime.datetime.now())
        result = resources.MssngrRouterStateCache(self, descr, work_dir, ttl=self.Parameters.resource_ttl)
        sdk2.ResourceData(result).ready()
