# -*- coding: utf-8 -*-

import os
import re
import shlex
import tempfile

import sandbox.projects.common.build.parameters as common_params
import sandbox.projects.mssngr.tasks.common.parameters as mssngr_parameters

from sandbox import sdk2
from sandbox.common import errors
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common.build.YaMake2 import ENV_VAR_PATTERN, VAULT_PATTERN
from sandbox.sandboxsdk.process import run_process


class ArgsParam(sdk2.parameters.String):
    name = 'program_args'
    description = 'Args to program'
    default_value = ''


class RunMssngrTask(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 2048
        disk_space = 2048

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        resource_type = mssngr_parameters.TaskType()
        args_param = ArgsParam()
        env_vars = common_params.EnvironmentVarsParam()

    def deref(self, s):
        def deref_vault(match):
            secret = sdk2.Vault.data(match.group('owner'), match.group('name'))

            if match.group('dst') == 'file':
                deref_path = tempfile.NamedTemporaryFile().name
                fu.write_file(deref_path, secret)
                return deref_path

            return secret

        if s.startswith('sec-'):
            s = sdk2.Vault.data(s)
        else:
            s = re.sub(VAULT_PATTERN, deref_vault, s)

        return s

    def get_env_vars(self):
        # XXX: copy-paste from YaMake2
        env_vars = self.Parameters.env_vars
        if env_vars and not re.match(ENV_VAR_PATTERN, env_vars):
            eh.check_failed("Incorrect 'Environment variables' parameter '{}'".format(env_vars))

        env_vars = {k: self.deref(v) for k, v in (x.split('=', 1) for x in shlex.split(env_vars))}

        if 'GSID' not in env_vars.keys() and 'GSID' in os.environ.keys():
            env_vars['GSID'] = os.getenv('GSID')

        return env_vars

    def on_execute(self):
        binary_resource = sdk2.Resource.find(
            type=self.Parameters.resource_type,
            attrs={"released": "stable"}
        ).first()

        if not binary_resource:
            raise errors.TaskFailure("Could not find binary")

        data = sdk2.ResourceData(binary_resource)

        run_process(
            cmd=[data.path] + self.Parameters.args_param.split(),
            shell=True,
            log_prefix="run",
            environment=self.get_env_vars(),
        )
