import logging
import os
import six

from textwrap import dedent

from sandbox import sdk2
from sandbox.projects.mssngr.zbp.common import (
    basic_task,
    helpers,
)
from sandbox.sdk2.helpers import subprocess

logger = logging.getLogger(__name__)


class MssngrZbpTask(basic_task.MssngrBasicTask):
    BIN_FILE_NAME = ""

    class Parameters(basic_task.MssngrBasicTask.Parameters):
        conf_args = sdk2.parameters.Dict("Bundle parameters")

    class MssngrOptions(object):
        # Download and use resource of this type.
        bundle_resource_type = None

        # Add to command command on execution phase.
        # If you want to generate command dynamically you need to override
        # get_cmd method of MssngrTask.
        cmd = []

        # Update common environment with this additional variables.
        # If you want to generate additional environment dynamically you
        # need to override get_additional_env method of CryptaTask.
        additional_env = None

        # Path to found binary, it can already be present on host, so
        # path will be different from base path
        path_to_binary = None

    def on_prepare(self):
        bundle = self.get_bundle_resource()
        bundle_path = helpers.download_resource(bundle)

        binary_name = os.path.basename(str(bundle_path))
        assert binary_name == self.BIN_FILE_NAME, \
            'Resource has binary `{}`, but expected `{}`'.format(binary_name, self.BIN_FILE_NAME)

        self.set_info(bundle)
        self.set_info(binary_name)

        self.MssngrOptions.path_to_binary = str(bundle_path)

        self.Context.bundle = dict(
            type=str(bundle.type),
            id=bundle.id,
        )
        self.Context.save()

    @sdk2.report("Report")
    def bundle_report(self):
        if not self.Context.bundle:
            return "No bundle information available"

        bundle = self.Context.bundle

        return dedent("""
            <ul>
                <li>Using <code>{bundle_type}</code> (<a href="/resource/{bundle_id}">#{bundle_id}</a>)</li>
            </ul>
        """).format(
            bundle_type=bundle['type'],
            bundle_id=bundle['id'],
        )

    def on_execute(self):
        with sdk2.helpers.ProcessLog(self, logger="mssngr_task") as pl:
            cmd = self.get_cmd()
            env = self.get_env()
            subprocess.check_call(cmd, shell=False, stdout=pl.stdout, stderr=pl.stderr, env=env)

    def on_save(self):
        self.Parameters.tags = [
            x for x in self.Parameters.tags
            if x.upper() not in helpers.get_possible_environment_tags()
        ]

        environment_tag = helpers.get_environment_tag(self.Parameters.environment)
        logger.info("Setting tag %s", environment_tag)
        self.Parameters.tags.append(environment_tag)

    def get_env(self):
        env = dict(os.environ)
        additional_env = self.get_additional_env()
        assert isinstance(additional_env, dict), "Additional environment must be instance of dict"
        env.update({k: v(self) if callable(v) else v for k, v in six.iteritems(additional_env)})
        return env

    def get_additional_env(self):
        return self.MssngrOptions.additional_env or {}

    def get_bundle_resource(self):
        if self.Parameters.override_basic_params and self.Parameters.overridden_bundle_resource:
            return self.Parameters.overridden_bundle_resource
        else:
            return helpers.get_last_released_resource(self.MssngrOptions.bundle_resource_type,
                                                      self.Parameters.environment)

    def conf_parameters(self):
        for (key, value) in (self.Parameters.conf_args or {}).items():
            yield "--{}".format(key)
            if value:
                yield str(value)

    def get_cmd(self):
        binary_path = self.MssngrOptions.path_to_binary
        cmd = [binary_path, ] + list(self.conf_parameters()) + self.MssngrOptions.cmd
        return cmd
