# coding=utf-8

import os
import logging

from sandbox.sdk2.vcs.git import Git
from sandbox.sdk2.helpers.process import subprocess
from sandbox import sdk2
from sandbox.projects.common.arcadia import sdk as arcadiasdk
from sandbox.sandboxsdk.environments import Xcode


class MssngrFrontElectronBuild(sdk2.Task):
    """A messenger task for build electron application"""

    class Requirements(sdk2.Task.Requirements):
        cores = 1

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Настройки сборки') as build_block:
            app_name = sdk2.parameters.String(
                'App name',
                default='team_chats',
                required=True,
                description='Какое приложение нужно собирать?',
                choices=[
                    ('Yandex Chats', 'chats'),
                    ('Yandex Team Chats', 'team_chats'),
                    ('Yandex Team Chats Nightly', 'team_chats_nightly')
                ]
            )

            with sdk2.parameters.CheckGroup('Платформы', required=True) as platforms:
                platforms.values.win = platforms.Value('Windows', checked=True)
                platforms.values.mac = platforms.Value('Mac OS', checked=True)
                platforms.values.linux = platforms.Value('Linux', checked=False)

            use_arc = sdk2.parameters.Bool(
                'Use arc',
                default=True,
                required=False,
                description='Клонировать через arc?'
            )

        with sdk2.parameters.Group('Настройки репозитория') as repo_block:
            with use_arc.value[True]:
                arc_path = sdk2.parameters.String(
                    "Arc path",
                    required=True,
                    default='frontend/services/messenger',
                    description='Путь в репозитории arc (напр. frontend/services/foo)'
                )

                arc_patch = sdk2.parameters.String(
                    'Arc patch',
                    description='Патч на код (text diff or rbtorrent or arc:id)'
                )

                # https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/common/arcadia/sdk.py?rev=r8685362#L3320
                arc_branch = sdk2.parameters.String(
                    'Arc branch',
                    description='Ветка для arcadia-arc:/# (https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/common/arcadia/sdk.py?rev=r8685362#L3320)'
                )

            with use_arc.value[False]:
                git_repo = sdk2.parameters.String(
                    "Owner/Repo",
                    required=True,
                    default='search-interfaces/frontend',
                    description='Репозиторий на github.yandex-team.ru (напр. search-interfaces/frontend)'
                )

                git_branch = sdk2.parameters.String(
                    'Build branch',
                    default='master',
                    required=True,
                    description='Имя ветки, из которой должна происходить сборка'
                )

        with sdk2.parameters.Group('Команды и дополнительные параметры') as commands_block:
            build_command = sdk2.parameters.String(
                "Build command",
                required=True,
                default='scripts/app/sandbox.sh',
                description='Команда запуска сборки'
            )

            vault_env = sdk2.parameters.Dict(
                "Vault items to put in the environment",
                description='Дополнительные переменные из секретницы, которые необходимы для запуска скрипта'
            )

            run_env = sdk2.parameters.Dict(
                "Custom variables to put in the environment",
                description='Дополнительные переменные окружения'
            )

            yav_env = sdk2.parameters.Dict(
                "Yav variables to put in the environment",
                description='Дополнительные переменные из yav'
            )

            custom_resources = sdk2.parameters.Dict(
                "Custom resources to upload",
                description='Ресурсы'
            )

    def get_env(self):
        logging.debug('Initializing environments: {}'.format(os.environ))

        command_env = os.environ.copy()

        command_env['SANDBOX_TASK_ID'] = str(self.id)
        command_env['SANDBOX_TASK_ROOT'] = '{}'.format(self.path())
        command_env['NVM_DIR'] = '{}/.nvm'.format(self.path())
        command_env['YENV'] = 'production'

        command_env["APP_NAME"] = self.Parameters.app_name
        command_env["APP_PLATFORMS"] = ','.join(platform.lower() for platform in self.Parameters.platforms)

        # Ключи для доступа в S3
        command_env["AWS_SECRET_ACCESS_KEY"] = sdk2.Vault.data("env.ROBOT_S3_SECRET_KEY")
        command_env["AWS_ACCESS_KEY_ID"] = sdk2.Vault.data("env.ROBOT_S3_ACCESS_KEY")

        # токен для статистики (для сборки)
        command_env["STATFACE_TOKEN"] = sdk2.Vault.data("env.ROBOT_MSSNGR_BUILDER_STATFACE_TOKEN")

        # только для сборки Mac OS
        command_env["MESSENGER_CSC_FILE_BASE64"] = sdk2.Vault.data("Q_APP_SIGN_KEY")
        command_env["MESSENGER_CSC_KEY_PASSWORD"] = sdk2.Vault.data("Q_APP_SIGN_PASSWORD")
        command_env["MESSENGER_SIGNER_OAUTH"] = sdk2.Vault.data("env.MESSENGER_SIGNER_OAUTH")

        for env_name, env_value in self.Parameters.run_env.items():
            command_env[env_name] = env_value

        for vault_item, env_name in self.Parameters.vault_env.items():
            command_env[env_name] = sdk2.Vault.data(vault_item)

        for resource_name, resource_id in self.Parameters.custom_resources.items():
            command_env['RESOURCE_' + resource_name.replace('RESOURCE_', '')] = self._download_resource(resource_id)

        for yav_name, yav_id in self.Parameters.yav_env.items():
            command_env['YAV_' + yav_name.replace('YAV_', '')] = sdk2.Vault.data(yav_id)

        logging.debug('Environments: {}'.format(command_env))

        return command_env

    def update_tags(self):
        tags = set(self.Parameters.tags)

        logging.debug('Base tags: {}'.format(list(tags)))

        tags = tags.union([
            self.Parameters.app_name.upper(),
        ])

        for platform in self.Parameters.platforms:
            tags = tags.union([
                platform.upper(),
            ])

        logging.debug('Update tags: {}'.format(list(tags)))

        self.Parameters.tags = list(tags)

    def git_clone(self):
        repo_url = "https://github.yandex-team.ru/{}.git".format(self.Parameters.git_repo)
        branch = self.Parameters.git_branch

        logging.debug('Start git clone from {}, branch {}'.format(
            repo_url,
            branch
        ))

        git = Git(repo_url)
        git.clone("repo", branch)

        logging.debug('End git clone')

    def get_arc_context(self):
        path = "arcadia:/arc/trunk/arcadia/"

        if self.Parameters.arc_branch:
            path = "arcadia-arc:/#" + self.Parameters.arc_branch

        logging.debug('Start arc checkout {}'.format(path))

        token = sdk2.Vault.data("MSSNGR_BUILDER_ARC_OAUTH")

        return arcadiasdk.mount_arc_path(
            path,
            arc_oauth_token=token,
            use_arc_instead_of_aapi=True
        )

    def apply_arc_patch(self, arcadia_src_dir):
        if self.Parameters.arc_patch:
            arcadiasdk.apply_patch(self, arcadia_src_dir, self.Parameters.arc_patch, self.path())
            logging.debug('Arc patch applied from {}'.format(self.Parameters.arc_patch))

    def on_enqueue(self):
        self.update_tags()

    def _download_resource(self, resource_id):
        logging.debug("Resource %d downloading", resource_id)
        resource = sdk2.Resource[resource_id]
        data = sdk2.ResourceData(resource)
        path = data.path  # See https://docs.python.org/dev/library/pathlib.html
        logging.info("Resource %d downloaded to %s", resource_id, path)

        return str(path)

    def pre_execute(self):
        logging.debug('Config Xcode env')
        Xcode.prepare_xcode()

    def on_execute(self):
        logging.debug('Self.path: {}'.format(self.path()))

        self.pre_execute()

        if self.Parameters.use_arc:
            arcadia_ctx = self.get_arc_context()
            with arcadia_ctx as arcadia_src_dir:
                self.Parameters.run_env['ARC_PATH'] = arcadia_src_dir
                self.apply_arc_patch(arcadia_src_dir)
                with sdk2.helpers.ProcessLog(self, logger='build') as pl:
                    subprocess.check_call(
                        self.Parameters.build_command,
                        env=self.get_env(),
                        cwd='{}/{}'.format(arcadia_src_dir, self.Parameters.arc_path),
                        shell=True,
                        stdout=pl.stdout,
                        stderr=pl.stderr,
                        close_fds=True
                    )
        else:
            self.git_clone()
            with sdk2.helpers.ProcessLog(self, logger='build') as pl:
                subprocess.check_call(
                    self.Parameters.build_command,
                    env=self.get_env(),
                    cwd="repo",
                    shell=True,
                    stdout=pl.stdout,
                    stderr=pl.stderr,
                    close_fds=True
                )
