# -*- coding: utf-8 -*-

from sandbox import sdk2
import json
import logging
import sandbox.projects.mt.spellchecker as spellchecker
import requests
from sandbox.projects.mt.components.spellchecker import SpellcheckerTesterTask

_logger = logging.getLogger(__name__)


def _load_ammo(queries_path):
    """
        :param queries_path: Queries for spellchecker
            (http://yandextank.readthedocs.io/en/latest/tutorial.html#uri-post-style)
    """

    headers = dict()
    input = open(queries_path)
    try:
        line = input.readline().strip()
        while line and line.startswith('[') and line.endswith(']'):
            name, value = line[1:-1].split(':')
            headers[name.strip()] = value.strip()
            line = input.readline().strip()
    except:
        input.close()
        raise

    def ammo_iter(line, input):
        try:
            while line:
                size_index = line.index(' ')
                body_size = int(line[:size_index])
                url = line[size_index + 1:]
                body = input.read(body_size)
                line = input.readline().strip()
                if not line:
                    line = input.readline().strip()
                yield url, body
        finally:
            input.close()

    return (headers, ammo_iter(line, input))


def _create_session(headers):
    retries = requests.packages.urllib3.util.retry.Retry(
            total=10,
            backoff_factor=0.1,
            status_forcelist=[500, 502, 503, 504])
    s = requests.Session()
    s.mount('http://', requests.adapters.HTTPAdapter(max_retries=retries))
    s.headers.update(headers)

    return s


def _query_spellchecker(queries, headers, out, daemon):
    url_base = 'http://{}:{}'.format(daemon.host, daemon.port)
    with _create_session(headers) as s:
        for url, body in queries:
            data = {"url": url}
            url_full = url_base + url
            try:
                r = s.post(url=url_full, data=body)
                r.raise_for_status()
                data["response"] = r.text
            except requests.exceptions.RequestException as e:
                data["error"] = str(e)
            json.dump(data, out)
            out.write('\n')

            if "error" in data:
                daemon.wait()


class CollectSpellcheckerResponses(SpellcheckerTesterTask):
    """
        Task for starting Spellchecker daemon and handling requests to it.
    """

    class Parameters(SpellcheckerTesterTask.Parameters):
        queries = sdk2.parameters.Resource('Spellchecker queries',
                    resource_type=spellchecker.SpellcheckerQueries,
                    required=True)

    def on_execute(self):
        queries_path = str(sdk2.ResourceData(self.Parameters.queries).path)
        queries_result = sdk2.ResourceData(spellchecker.SpellcheckerQueriesResult(self, 'Queries result',
                    'queries_result.txt'))

        with self.StartDaemon() as daemon:
            _logger.info('started')
            headers, queries = _load_ammo(queries_path)
            with open(str(queries_result.path), 'w') as output:
                _query_spellchecker(queries, headers, output, daemon)
            _logger.info("requests finished")
        _logger.info("stopped")
