# -*- coding: utf-8 -*-

import os
import logging
import sandbox.projects.mt.spellchecker as spellchecker
from sandbox import sdk2
import sandbox.projects.common.search.components.component as components_common

_logger = logging.getLogger(__name__)


class SpellcheckerDaemonWrapper(
        components_common.ProcessComponentMixin,
        components_common.WaitUrlComponentMixin,
        components_common.Component):
    """
        Wrapper over 'spellchecker daemon'
    """

    def __init__(self, snapshot_dir, port, threads, mlock, no_precharge,
                start_timeout, shutdown_timeout, daemon_path=None, data_dir=None,
                server_config_path=None, misspell_config_path=None, logs_dir=None,
                ping_url='/misspell/check'):
        """
            :param snapshot_dir: directory that consists default 'bin', 'config' and 'data' directories
            :param logs_dir: directory to write logs to
            :param port: port for daemon to listen to
            :param threads: count of threads
            :param mlock: lock data files in memory (for production mode only)
            :param no_precharge: disable precharge (for fast loading)
            :param start_timeout: daemon start timeout (in seconds)
            :param shutdown_timeout: daemon shutdown timeout (in seconds)
            :param daemon_path: path to 'spellchecker' executable
            :param data_dir: data directory
            :param server_config_path: server config file (e.g. conf/server.cfg)
            :param misspell_config: misspell config file (e.g. conf/misspell.json)
            :param logs_dir: directory to write logs to
            :param ping_url: relative url to ping service
        """

        _logger.info("Current working direcory: %s", os.getcwd())
        assert os.path.isdir(snapshot_dir)
        assert os.access(snapshot_dir, os.R_OK)  # is directory readable

        assert isinstance(threads, (int, long))

        assert isinstance(mlock, bool)

        assert isinstance(no_precharge, bool)

        assert isinstance(start_timeout, (int, long))

        assert isinstance(shutdown_timeout, (int, long))

        if daemon_path is None:
            daemon_path = os.path.join(snapshot_dir, "bin", "spellchecker")

        if data_dir is None:
            data_dir = os.path.join(snapshot_dir, "data")
        assert os.path.isdir(data_dir)
        assert os.access(data_dir, os.R_OK)  # is directory readable
        os.symlink(data_dir, "data")

        conf_dir = os.path.join(snapshot_dir, "conf")
        assert os.path.isdir(conf_dir)
        assert os.access(conf_dir, os.R_OK)  # is directory readable
#os.symlink(conf_dir, "conf")
        os.mkdir("conf")

        if misspell_config_path is None:
            misspell_config_path = os.path.join(conf_dir, "misspell.json")
        assert os.path.isfile(misspell_config_path)
        assert os.access(misspell_config_path, os.R_OK)  # is file readable
        os.symlink(misspell_config_path, "conf/misspell.json")

        if server_config_path is None:
            server_config_path = os.path.join(conf_dir, "server.cfg")
        assert os.path.isfile(server_config_path)
        assert os.access(server_config_path, os.R_OK)  # is file readable

        if logs_dir is not None:
            assert os.path.isdir(logs_dir)
            assert os.access(logs_dir, os.W_OK)  # is directory writable
            os.symlink(logs_dir, "log")

        args = [daemon_path,
                '-c', server_config_path,
                '-n', threads,
                '-p', port]
        if mlock:
            args.append('--mlock')
        if no_precharge:
            args.append('--no-precharge')

        self.host = "localhost"
        self.port = port
        components_common.ProcessComponentMixin.__init__(
                self,
                args=args,
                shutdown_timeout=shutdown_timeout,
                shell=True)
        components_common.WaitUrlComponentMixin.__init__(
                self,
                url="http://{}:{}{}".format(self.host, self.port, ping_url),
                wait_timeout=start_timeout)


class SpellcheckerTesterTask(sdk2.Task):
    """
        Task for starting Spellchecker daemon and handling requests to it.
    """

    class Parameters(sdk2.Task.Parameters):
        # custom paremeters
        with sdk2.parameters.Group('Group of parameters for spellchecker daemon'):
            snapshot = sdk2.parameters.Resource(
                    'Spellchecker snapshot',
                    resource_type=spellchecker.SpellcheckerSnapshot,
                    required=True)
            executable = sdk2.parameters.Resource(
                    'Spellchecker daemon',
                    resource_type=spellchecker.SpellcheckerExecutable)
            data = sdk2.parameters.Resource(
                    'Spellchecker data',
                    resource_type=spellchecker.SpellcheckerData)
            misspell_config = sdk2.parameters.Resource(
                    'Spellchecker config (e.g. conf/misspell.ini)',
                    resource_type=spellchecker.SpellcheckerMisspellConfig)
            server_config = sdk2.parameters.Resource(
                    'Spellchecker server config (e.g. conf/server.cfg)',
                    resource_type=spellchecker.SpellcheckerServerConfig)
            threads = sdk2.parameters.Integer(
                    'Spellchecker threads count',
                    default=1)
            mlock = sdk2.parameters.Bool(
                    'Spellchecker lock data files in memory (for production mode only)',
                    default=False)
            no_precharge = sdk2.parameters.Bool(
                    'Spellchecker disable precharge (for fast loading)',
                    default=False)
            start_timeout = sdk2.parameters.Integer(
                    'Spellchecker start timeout in seconds',
                    default=60 * 60)
            shutdown_timeout = sdk2.parameters.Integer(
                    'Spellchecker shutdown timeout in seconds',
                    default=2 * 60)

    def StartDaemon(self):
        snapshot_dir = str(sdk2.ResourceData(self.Parameters.snapshot).path)
        daemon_path = self.Parameters.executable
        daemon_path = None if daemon_path is None else str(sdk2.ResourceData(daemon_path).path)
        data_dir = self.Parameters.data
        data_dir = None if data_dir is None else str(sdk2.ResourceData(data_dir).path)
        m_config_path = self.Parameters.misspell_config
        m_config_path = None if m_config_path is None else str(sdk2.ResourceData(m_config_path).path)
        s_config_path = self.Parameters.server_config
        s_config_path = None if s_config_path is None else str(sdk2.ResourceData(s_config_path).path)

        logs_dir_data = sdk2.ResourceData(spellchecker.SpellcheckerLogsDirectory(
                                            self,
                                            'Spellchecker logs directory',
                                            'spellchecker_logs'))
        logs_dir_data.path.mkdir()
        logs_dir_path = str(logs_dir_data.path)

        port = components_common.try_get_free_port(15)
        daemon = SpellcheckerDaemonWrapper(snapshot_dir, port, self.Parameters.threads,
                self.Parameters.mlock, self.Parameters.no_precharge, self.Parameters.start_timeout,
                self.Parameters.shutdown_timeout, daemon_path, data_dir, s_config_path,
                m_config_path, logs_dir_path)

        return daemon
