# -*- coding: utf-8 -*-

import datetime
import logging

from io import open

import sandbox.common.types.client as ctc
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.resource_types import OTHER_RESOURCE


def write_table_into_stream(table, stream):
    """
        Write some (possibly unicode, possibly not) text into a stream
        without storing the whole table in memory
    """
    # for row in get_iterator(table):
    for row in table.get_iterator():
        text = "\t".join(row) + "\n"
        if not isinstance(text, unicode):
            try:
                text = unicode(text, "utf-8")
            except UnicodeError:
                raise Exception(text)
        stream.write(text)


def save_result_into_file(request, file_path):
    from yql.config import config
    num_tables = len(list(request.get_results()))
    assert num_tables == 1, \
        "returned {} != 1 tables".format(num_tables)
    for table in request.get_results():
        config.no_column_names = True
        config.no_column_types = True
        config.encoding = "utf-8"
        with open(file_path, "w", encoding="utf-8") as out_file:
            write_table_into_stream(
                table,
                out_file,
            )


class MkmapResource(sdk2.Resource):
    """
        A type for the result .map file
    """
    __default_attribute__ = sdk2.parameters.String


class ExcludeCompaniesParameters(sdk2.Task.Parameters):
    with sdk2.parameters.Group("Vault") as vault:
        VAULT_YQL_OWNER = sdk2.parameters.String("Vault YQL token owner", required=True)
        VAULT_YQL_TOKEN = sdk2.parameters.String(
            "Name of vault YQL token",
            required=True
        )
    sql_script = sdk2.parameters.Resource("script to run", resource_type=OTHER_RESOURCE, required=True)
    # libmsp.so -- 576834364
    attrs_companies = sdk2.parameters.Resource(
        "attrs.companies resource",
        required=True
    )
    mkmap_binary = sdk2.parameters.Resource(
        "mkmap binary",
        required=True
    )
    output_map_path = sdk2.parameters.String(
        "result .map file",
        default="out.map",
        required=True
    )
    resource_attrs = sdk2.parameters.String(
        "Set attrs to resources (e.g.: attr1=v1, attr2=v2)",
        default="",
        required=False
    )


class ExcludeCompanies(sdk2.Task):
    """ Make a .map file with oddly names companies """

    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.GENERIC & ctc.Tag.Group.LINUX

    class Parameters(ExcludeCompaniesParameters):
        pass

    def on_execute(self):
        from yql.api.v1.client import YqlClient

        with sdk2.helpers.ProcessLog(
            self, logger=logging.getLogger('exclude-companies-log')
        ) as ec_log:
            yql_token = sdk2.Vault.data(
                self.Parameters.VAULT_YQL_OWNER,
                self.Parameters.VAULT_YQL_TOKEN
            )
            client = YqlClient(token=yql_token)
            with open(
                str(sdk2.ResourceData(self.Parameters.sql_script).path)
            ) as f:
                # resource 583983056
                script = f.read().replace("@@date", str(datetime.date.today()))
            request = client.query(script)
            request.run()

            tmp_data_path = "out.tsv"
            save_result_into_file(request, tmp_data_path)

            mkmap_binary = self.Parameters.mkmap_binary
            sp.Popen(
                [
                    str(sdk2.ResourceData(mkmap_binary).path),
                    "create",
                    "-a",
                    str(
                        sdk2.ResourceData(self.Parameters.attrs_companies).path
                    ),
                    "-w",
                    str(self.Parameters.output_map_path),
                    tmp_data_path,
                ],
                stdout=ec_log.stdout,
                stderr=ec_log.stderr,
            ).wait()

            resource_attrs = self.Parameters.resource_attrs.replace(" ", "").split(",")
            resource_attrs = dict(x.split("=") for x in resource_attrs)
            date = datetime.datetime.now()
            descr = "[{date:%Y-%m-%d_%H-%M-%S}] mkmap file with the correct spellings".format(date=date)
            if "type" in resource_attrs:
                descr += " of some " + resource_attrs["type"]
            if "lang" in resource_attrs:
                descr += " [lang = " + resource_attrs["lang"] + "]"
            map_file_resource = MkmapResource(
                self,
                description=descr,
                path=str(self.Parameters.output_map_path),
                **resource_attrs
            )
            map_file_data = sdk2.ResourceData(map_file_resource)
            map_file_data.ready()
