import json
import logging
import os

from pathlib2 import Path
from sandbox import common, sdk2
from sandbox.projects.common import binary_task, task_env
from sandbox.projects.common.vcs.arc import Arc, ArcCommandFailed
from sandbox.projects.mt.make.util import mount_arc_with_retries

import sandbox.common.types.client as ctc
import sandbox.common.types.notification as ctn
import sandbox.common.types.task as ctt
from sandbox.common.types import misc as ctm


class DownloadOpus(binary_task.LastBinaryTaskRelease, sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)

        description = "Download latest OPUS corpora"
        owner = "MT"

        notifications = [
            sdk2.Notification(
                statuses=[
                    ctt.Status.FAILURE,
                    ctt.Status.EXCEPTION,
                    ctt.Status.TIMEOUT
                ],
                recipients=["dronte@yandex-team.ru"],
                transport=ctn.Transport.EMAIL
            )
        ]

        pull_request_id = sdk2.parameters.Integer("Id of pull request to run code from (empty for trunk)")
        direction = sdk2.parameters.String("Direction to download corpora for", required=True)
        secret = sdk2.parameters.YavSecret("YAV secret identifier (with optional version)")

        #with sdk2.parameters.Output():
        #    workflow_id = sdk2.parameters.String("Id of started workflow on Nirvana")
        #    workflow_instance_id = sdk2.parameters.String("Id of started workflow instance on Nirvana")

    class Requirements(task_env.BuildLinuxRequirements):
        dns = ctm.DnsType.DNS64

    def on_execute(self):
        res = self.build_and_run()
        #self.Parameters.workflow_id = res['workflow_id']
        #self.Parameters.workflow_instance_id = res['workflow_instance_id']

    def build_and_run(self):
        binary_dir = os.path.join("dict", "mt", "make", "tools", "download_opus")

        arc_token = self.Parameters.secret.data()['arc-token']
        arc = Arc(arc_oauth_token=arc_token)

        nirvana_token = self.Parameters.secret.data()['nirvana-token']
        nirvana_quota = 'mt'

        home_path = Path(os.getcwd())

        env = {
            'HOME': str(home_path),
            'YA_TOKEN': self.Parameters.secret.data()['ya-token'],
            'YT_TOKEN': self.Parameters.secret.data()['yt-token']
        }

        with mount_arc_with_retries(arc) as arc_mount_path:
            if self.Parameters.pull_request_id:
                arc.pr_checkout(arc_mount_path, self.Parameters.pull_request_id)

            with sdk2.helpers.ProcessLog(self, logger='build') as pl:
                sdk2.helpers.subprocess.check_call(
                    [
                        os.path.join(arc_mount_path, "ya"),
                        "make",
                        "--yt-store", "-r",
                        binary_dir
                    ],
                    cwd=arc_mount_path, env=env,
                    stdout=pl.stdout, stderr=pl.stderr,
                )

            with sdk2.helpers.ProcessLog(self, logger='run') as pl:
                sdk2.helpers.subprocess.check_call(
                    [
                        os.path.join(arc_mount_path, binary_dir, os.path.basename(binary_dir)),
                        self.Parameters.direction,
                        "--print-db-config",
                        "--corpus-names", "all-latest",
                        "--arcadia-root", arc_mount_path
                    ],
                    cwd=str(home_path), env=env,
                    stdout=pl.stdout, stderr=pl.stderr,
                )
