import contextlib
import json
import os
import requests
import yaml

from pathlib2 import Path
from sandbox import sdk2
from sandbox.projects.common.vcs.arc import Arc, ArcCommandFailed


def mount_arc_with_retries(arc=None, path=None, changeset=None, arc_token=None, num_retries=3, fetch_all=False, **kwargs):
    if arc is None:
        assert arc_token is not None
        arc = Arc(arc_oauth_token=arc_token)
    else:
        assert arc_token is None

    while True:
        try:
            return arc.mount_path(path, changeset, fetch_all=fetch_all, **kwargs)
        except ArcCommandFailed as e:
            if num_retries <= 0:
                raise e
            else:
                num_retries -= 1


class MountArc:
    def __init__(self, branch, arc_token, create_branch_if_not_exists=True):
        self.branch = branch
        self.arc_token = arc_token
        self.create_branch_if_not_exists = create_branch_if_not_exists
        self.commits_hashes = []

    @property
    def last_commit_hash(self):
        if len(self.commits_hashes) == 0:
            return None
        return self.commits_hashes[-1]

    def checkout(self, branch, create_branch_if_not_exists=True, fail_if_branch_exists=False):
        try:
            self.arc.checkout(self.mount_path, branch)
            if fail_if_branch_exists:
                raise RuntimeError('Branch ' + branch + ' already exists')
        except ArcCommandFailed:
            if create_branch_if_not_exists:
                self.arc.checkout(self.mount_path,
                                  branch,
                                  create_branch=True)
            else:
                raise
        self.branch = branch

    def __enter__(self):
        self.arc = Arc(arc_oauth_token=self.arc_token)
        self.mount_context = mount_arc_with_retries(self.arc)
        self.mount_path = self.mount_context.__enter__()
        self.checkout(self.branch, create_branch_if_not_exists=self.create_branch_if_not_exists)
        self.merge_base = self.arc.get_merge_base(self.mount_path)
        self.on_mount_head = self.arc.get_merge_base(self.mount_path, 'HEAD', 'HEAD')
        return self

    @contextlib.contextmanager
    def commit(self, message, push=True, allow_empty=False):
        yield None
        # thrown exception canceles the commit
        try:
            arc_message = self.arc.commit(
                self.mount_path,
                message,
                all_changes=True)
        except ArcCommandFailed as e:
            if not allow_empty:
                raise
        else:
            commit_hash = arc_message.split(']')[0].split()[1]
            self.commits_hashes.append(commit_hash)
        finally:
            if push:
                self.arc.push(self.mount_path, upstream=self.branch)

    def create_pr(self, message):
        self.arc.pr_create(self.mount_path, publish=True, auto=True, message=message)

    def __exit__(self, exc_type, exc_value, traceback):
        self.mount_context.__exit__(exc_type, exc_value, traceback)


@contextlib.contextmanager
def push_arcadia_commit(branch, message, arc_token, create_branch_if_not_exists=True, create_pr=False, pr_message=None):
    with MountArc(branch=branch,
                  arc_token=arc_token,
                  create_branch_if_not_exists=create_branch_if_not_exists
                  ) as arc:
        with arc.commit(message=message):
            yield arc.mount_path
        if create_pr:
            arc.create_pr(message=pr_message)




def post_pull_request_comment(pull_request_id, comment, arcanum_token):
    res = requests.post(
        "https://a.yandex-team.ru/api/v1/pull-requests/%d/comments" % pull_request_id,
        json={'content': comment},
        headers={'Authorization': 'OAuth %s' % arcanum_token}
    )
    res.raise_for_status()


def apply_mtdata_updates(arc_root, mtdata_updates):
    with open(os.path.join(arc_root, 'dict', 'mt', 'data.yaml'), 'rb+') as f:
        data = yaml.safe_load(f) or {}
        data.update(yaml.safe_load(mtdata_updates))
        data = dict(sorted(data.items()))  # Sort data by key

        # Rewrite file content
        f.seek(0)
        f.write(yaml.dump(data, allow_unicode=True, default_flow_style=False))
        f.truncate()


def run_mt_make_vh_tool(tool, args, arcadia_path, secrets, quota='mt', mr_account='mt', work_dir=None, name=None):
    home_path = Path.home()

    vhrc_path = home_path.joinpath(".vhrc")
    vhrc_path.write_text('\n'.join([
        "--oauth-token=%s" % secrets['nirvana-token'],
        "--quota=%s" % quota,
        "--mr-account=%s" % mr_account,
        "--arcadia-root=%s" % arcadia_path,
        "--api-retries=25",
        "--api-retry-delay=2",
        "--api-retry-delay-multiplier=1.5",
        "--api-retry-methods=all",
    ]))
    vhrc_path.chmod(0o400)

    wi_path = home_path.joinpath("wi.json")

    run_mt_make_tool(
        tool,
        args + ["--write-workflow-info", str(wi_path)],
        arcadia_path=arcadia_path,
        secrets=secrets,
        work_dir=work_dir,
        name=name,
    )

    return json.loads(wi_path.read_text())


def run_mt_make_tool(tool, args, arcadia_path, secrets, work_dir=None, name=None):
    if work_dir is None:
        work_dir = Path.home()

    env = {
        'HOME': str(Path.home()),
        'YA_TOKEN': secrets['ya-token'],
        'YT_TOKEN': secrets['yt-token'],
        'YT_PROXY': 'hahn'
    }

    if name is None:
        name = os.path.basename(tool)

    with sdk2.helpers.ProcessLog(logger='%s-build' % name) as pl:
        sdk2.helpers.subprocess.check_call(
            [
                os.path.join(arcadia_path, "ya"),
                "make",
                "--yt-store", "-r",
                os.path.dirname(tool)
            ],
            cwd=arcadia_path, env=env,
            stdout=pl.stdout, stderr=pl.stderr,
        )

    with sdk2.helpers.ProcessLog(logger='%s-run' % name) as pl:
        sdk2.helpers.subprocess.check_call(
            [os.path.join(arcadia_path, tool)] + args,
            cwd=str(work_dir), env=env,
            stdout=pl.stdout, stderr=pl.stderr,
        )


def run_mt_make_test(tool, args, arcadia_path, secrets):
    home_path = Path.home()

    env = {
        'HOME': str(home_path),
        'YA_TOKEN': secrets['ya-token'],
        'YT_TOKEN': secrets['yt-token'],
        'YT_PROXY': 'hahn'
    }
    name = tool.replace('/', '.')

    with sdk2.helpers.ProcessLog(logger='%s-test' % name) as pl:
        sdk2.helpers.subprocess.check_call(
            [
                os.path.join(arcadia_path, "ya"),
                "make",
                "--yt-store", "-ttt",
                tool,
            ] + args,
            cwd=arcadia_path, env=env,
            stdout=pl.stdout, stderr=pl.stderr,
        )
