# coding=utf-8

from __future__ import absolute_import

import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr
import datetime
from sandbox import projects
import sandbox.projects.mt.util.fs as mt_fs
from sandbox import sdk2

from sandbox.projects.common.nanny import nanny


class MakeSpellcheckerSnapshotResourceParameters(sdk2.Task.Parameters):
    with sdk2.parameters.Group("Snapshot build parameters"):
        base_snapshot_r = sdk2.parameters.Resource(
            'Snapshot to modify (default: take last released resource)',
            resource_type=projects.mt.spellchecker.SpellcheckerSnapshot, )
        exclude_ru_lm = sdk2.parameters.Bool('Exclude Russian lm')
        exclude_uk_lm = sdk2.parameters.Bool('Exclude Ukranian lm')


def _get_snapshot_data(resource_id=None):
    if resource_id is None:
        resource_id = projects.mt.spellchecker.SpellcheckerSnapshot.find(
            state=ctr.State.READY,
            attrs={'released': 'stable'}, ).first()

    return sdk2.ResourceData(resource_id)


class MakeSpellcheckerSnapshotResource(nanny.ReleaseToNannyTask2, sdk2.Task):
    """ Make spellchecker snapshot resource from parts """

    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.GENERIC & ctc.Tag.Group.LINUX
        disk_space = 200 * 1024  # 200 GB

    class Parameters(MakeSpellcheckerSnapshotResourceParameters):
        pass

    def on_release(self, params):
        nanny.ReleaseToNannyTask2.on_release(self, params)
        sdk2.Task.on_release(self, params)
        self.mark_released_resources(params['release_status'], ttl=30)

    def on_execute(self):
        date = datetime.datetime.now()

        base_snapshot_data = _get_snapshot_data(self.Parameters.base_snapshot_r)

        snapshot_descr = '[new_pipeline] Spellchecker snapshot at {date:%Y-%m-%d_%H-%M-%S}'.format(date=date)
        snapshot_resource = projects.mt.spellchecker.SpellcheckerSnapshot(
            self,
            description=snapshot_descr,
            path='spellchecker',
            ttl=30, )
        snapshot_data = sdk2.ResourceData(snapshot_resource)
        snapshot_data.path.mkdir(parents=True, exist_ok=True)

        mt_fs.copy_tree_using_hardlinks(
            src=str(base_snapshot_data.path),
            dst=str(snapshot_data.path),
            override=True, )

        if not self.Parameters.exclude_ru_lm:
            mt_fs.override_file_with_resource(
                resource_type=projects.mt.spellchecker.SpellcheckerSnapshotPart,
                resource_attributes={'lang': 'ru', 'released': 'stable'},
                dst_rel_path='data/all/lm/word-3grams.trie',
                dst_dir=str(snapshot_data.path), )

        if not self.Parameters.exclude_uk_lm:
            mt_fs.override_file_with_resource(
                resource_type=projects.mt.spellchecker.SpellcheckerSnapshotPart,
                resource_attributes={'lang': 'uk', 'released': 'stable'},
                dst_rel_path='data/langs/uk/lm/word-3grams.trie',
                dst_dir=str(snapshot_data.path), )

        snapshot_data.ready()
