# coding=utf-8

import logging
import re

import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.projects.mt.product import TranslateWebLxcContainer
from sandbox.projects.mt.util.arc_mixin import ArcMixin
from sandbox.projects.mt.util.mt_web import MtWebMixin
from sandbox.projects.mt.util.task_profiler import TaskProfilerMixin

DEV_HOST = 'https://st.test.yandex-team.ru/'
PROD_HOST = 'https://st.yandex-team.ru/'


class ReleaseTranslateTicketTask(TaskProfilerMixin, ArcMixin, MtWebMixin, sdk2.Task):
    """
    Create release ticket task
    """

    class Requirements(sdk2.Requirements):
        cores = 2
        ram = 8000
        disk_space = 16000
        dns = ctm.DnsType.DNS64

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Output():
            release_ticket = sdk2.parameters.String('release ticket')
            tickets_list = sdk2.parameters.String('changelog')

        with sdk2.parameters.Group('Arc params') as arc_group:
            arc_params = ArcMixin.ArcParams()

        with sdk2.parameters.Group('Commit params', collapse=True) as commit_group:
            start_commit = sdk2.parameters.String('start_commit')
            end_commit = sdk2.parameters.String('end_commit')

        with sdk2.parameters.Group('Tracker params', collapse=True) as tracker_group:
            dry_run = sdk2.parameters.Bool('Dry run')
            tracker_token = sdk2.parameters.YavSecret(
                'Tracker token',
                default='sec-01daxjack0ee8kdeqsb9xc5kzm'
            )
            version = sdk2.parameters.String('Release version', required=True)
            assignee = sdk2.parameters.String('Assignee', required=True)
            followers = sdk2.parameters.List('Followers')
            tickets = sdk2.parameters.List('Ticket keys')
            dev = sdk2.parameters.Bool('Use dev api', default_value=False)

        with sdk2.parameters.Group('Caching params', collapse=True) as caching_params_group:
            caching_params = MtWebMixin.CachingParams()

        with sdk2.parameters.Group('Advanced task params', collapse=True) as advanced_task_params_group:
            process_params = MtWebMixin.ProcessParams()

        _container = sdk2.parameters.Container(
            label='Translate container',
            resource_type=TranslateWebLxcContainer,
            platform='linux_ubuntu_16.04_xenial',
            required=True,
        )

    def on_execute(self):
        super(ReleaseTranslateTicketTask, self).on_execute()
        self.init_mt_web(task=self)
        self.init_arc(task=self)
        self.init_task_profiler(task=self)

        try:
            with self.profiler.action('Mount Arc'):
                self.mount_arc(fetch_all=True)

            self.run_main_actions()
        finally:
            with self.profiler.action('Unmount Arc'):
                self.unmount_arc()

    def run_main_actions(self):
        with self.profiler.action('Setup venv'):
            self.setup_venv()

        with self.profiler.action('Get changelog & create release ticket'):
            self.run_tracker_action()

    def run_tracker_action(self):
        changelog = self.create_changelog()
        release_ticket_key = self.create_release_ticket(changelog)
        logging.info('release ticket is {}'.format(release_ticket_key))

    def create_release_ticket(self, changelog):
        if self.Parameters.dry_run:
            return 'MT-1'

        logging.info('create release ticket')
        args = [
            'ci-create-release-ticket',
            '--version=' + str(self.Parameters.version),
            '--tickets=' + ','.join(changelog),
            '--assignee=' + str(self.Parameters.assignee),
            '--followers=' + ','.join(self.Parameters.followers),
            '--token=' + self.Parameters.tracker_token.data()['secret']
        ]
        if self.Parameters.dev:
            args.append('--dev')
        out = self.run_project_method(*args)
        key = self.get_var_from_method_out(out, 'release_ticket_key')
        self.Parameters.release_ticket = ticket_to_link(key, self.Parameters.dev)
        return key

    def create_changelog(self):
        changelog = self._get_changelog_internal()
        logging.info('changelog - {}'.format(changelog))
        self.Parameters.tickets_list = tickets_to_links(changelog)
        return changelog

    def _get_changelog_internal(self):
        if len(list(self.Parameters.tickets)) > 0:
            logging.info('get changelog from input params')
            return self._get_changelog_from_params()
        else:
            logging.info('get changelog from arc')
            return self._get_changelog_from_arc()

    def _get_changelog_from_params(self):
        return self.Parameters.tickets

    def _get_changelog_from_arc(self):
        commits = self.arc.log(
            self.repo_path,
            path='dict/mt/web',
            start_commit=self.Parameters.start_commit,
            end_commit=self.Parameters.end_commit,
            as_dict=True
        )
        return find_tickets(commits)


def find_tickets(commits):
    result = set()
    for commit in commits:
        tickets = extract_ticket(commit['message'])
        if len(tickets) > 0:
            for t in tickets:
                result.add(t)
    return list(result)


def extract_ticket(msg):
    return re.findall(r'MT-\d+', msg)


def ticket_to_link(ticket, use_dev_host):
    host = DEV_HOST if use_dev_host else PROD_HOST
    return host + ticket


def tickets_to_links(tickets, use_dev_host=False):
    return '\n'.join(map(lambda ticket: ticket_to_link(ticket, use_dev_host), tickets))
